/*= Instrument bp2329 =====================================================*/
/* LabWindows 2.3 Instrument Driver                                        */
/* Original Release: bp2329.c   V1.0                     Date: 08.04.97/MW */
/*          Release: bp2329.c   V1.1                     Date: 29.10.97/MW */
/*          Release: bp2329.c   V1.2                     Date: 17.09.03/MT */
/*			  Changes: Com-port extended to COM8 in function 			   */
/*					   bp2329_init_serial.								   */
/* By: M.Westermann                                                        */
/* Originally written in C                                                 */
/*=========================================================================*/
#include <rs232.h>
#include <formatio.h>
#include <utility.h>
#include <gpib.h>
#include <ansi_c.h>
#include "bp2329.h"



/*= STATIC VARIABLES ======================================================*/
static int interface;                     /* 0 = GPIB; 1 = Serial*/
static int setup_first ;                  /* remark the first initial*/
static double std_timeout;                /* remark the user timeout*/
static int instr_cnt;
static int bp2329_err;



/*= INSTRUMENT-DEPENDENT COMMAND ARRAYS ===================================*/
/* INSERT COMMAND ARRAY DECLARATIONS HERE. SEE THE BOTTOM OF THIS FILE FOR */
/* FURTHER EXPLANATION.                                                    */
/*=========================================================================*/

static int address[bp2329_MAX_INSTR + 1];  /*GPIB-Adress*/
static int bd[bp2329_MAX_INSTR + 1];       /*Gpib Board*/
static char cmd[bp2329_MAX_CMD+1];
static char buf[bp2329_MAX_CMD+1];
static char out_buf[bp2329_MAX_CMD+1];       /* buffer for serial send */
static char in_buf[bp2329_MAX_CMD+1];        /* buffer for serial receive */
static int port_addr[8] ;                    /* RS232-Port Array*/
static int int_level[4] ;                    /* RS232-Interrupt-Level Array*/
static int dev_port[bp2329_MAX_INSTR+1];       /* Device Port*/

static char far* sense_aver_ctrl[2];
static char far* sense_loading[2];
static char far* sense_range[8];
static char far* sense_lower_range[8];
static char far* sense_upper_range[8];    
   
static char far* sense_meas_mode[2];
static char far* sense_limit_20mv[2];
static char far* sense_meas_sequence[4];
static char far* sense_num_of_cycles[4];
static char far* display_mode[3];
static char far* display_resolution[2];
static char far* calc_limit_fault[2];
static char far* sense_tcomp[4];

static char far* cal_state[2];
static char far* cal_range[7];
static char far* cal_20mV_mode[2];
static char far* cal_voltage_mode[2];
static char far* cal_pt100_mode[2];

/*= UTILITY ROUTINES ======================================================*/

int _VI_FUNC  bp2329_invalid_integer_range (int, int, int, int);
int _VI_FUNC  bp2329_invalid_real_range (double, double, double, int);
int _VI_FUNC  bp2329_open_instr_gpib(int );
int _VI_FUNC  bp2329_device_closed_gpib (int);
int _VI_FUNC  bp2329_close_instr_gpib (int);
int _VI_FUNC  bp2329_read_msg_gpib (int,char*,int);
int _VI_FUNC  bp2329_write_msg_gpib (int,char *, int);
void bp2329_setup_arrays (void);
int _VI_FUNC  bp2329_get_status(int);
int _VI_FUNC  bp2329_send_rs232(int,char*);
int _VI_FUNC  bp2329_receive_rs232(int,char*,char*);
int _VI_FUNC  bp2329_receive_rs232_message(int ,char* );
int _VI_FUNC  bp2329_close_instr_serial (int);

int _VI_FUNC  bp2329_open_instr_serial (int,long,double);
int _VI_FUNC  bp2329_read_oper_event_reg(int instrID,int* status);
int _VI_FUNC  bp2329_read_quest_fres_reg(int instrID,int* status);
int _VI_FUNC  bp2329_read_quest_reg(int instrID,int* status);
int _VI_FUNC  bp2329_read_std_event_reg(int instrID,int* status);
int   get_range_number(char * range_txt);
double  convert_string_to_value(char * string);



/*=========================================================================*/
/* This function opens the instrument, queries the instrument for its ID,  */
/* and initializes the instrument to a known state.                        */
/*=========================================================================*/
int _VI_FUNC bp2329_init_gpib(int addr,int time_out,int reset,char *model,int* instrID)
{
    int ID;
    bp2329_err = 0;
    bp2329_setup_arrays ();
    
    interface = bp2329_GPIB;            /* GPIB */

    if (bp2329_invalid_integer_range (addr, 0, 30,  -1) != 0)
     return bp2329_err;
    if (bp2329_invalid_integer_range (reset, 0, 1,  -2) != 0)
     return bp2329_err;

    ID = bp2329_open_instr_gpib (addr);
    if (ID <= 0)
      return bp2329_err;

    ibtmo (bd[ID], time_out);
    
    /*  terminate read on LF  */
    if (ibeos (bd[ID], 0x40a) < 0)
    {
     bp2329_err = 237;
     return bp2329_err;
    }
    
	ibconfig (bd[ID],IbcREADDR ,1);   /* Repeat Addressing */
	
    if (reset == 1)
    {
     if ((ibclr (bd[ID]) & (int)0x8000 != 0) != 0)
     {
      bp2329_close_instr_gpib (ID);
      bp2329_err = 224;
      return bp2329_err;
     }

     if (bp2329_send_scpi (ID,"*RST") != 0)
     {
      bp2329_close_instr_gpib (ID);
      return bp2329_err;
     }
     if (bp2329_send_scpi (ID,"*CLS") != 0)
     {
      bp2329_close_instr_gpib (ID);
      return bp2329_err;
     }
     
    }

    if (bp2329_receive_scpi(ID,"*IDN?",model) != 0)
    {
     bp2329_close_instr_gpib (ID);
     return bp2329_err;
    }
    

    
    Fmt(buf,"STAT:OPER:ENAB %i",bp2329_EOC);/* EOC als Maske Freigeben*/
    if (bp2329_send_scpi (ID,buf) != 0)
    {
     bp2329_close_instr_gpib (ID);
     return bp2329_err;
    }
	/*------- Temperatur und Widerstand Fehler freigeben------ */    
    Fmt(buf,"STAT:QUES:ENAB %i",(bp2329_TEMP_ERROR + bp2329_FRES_ERROR));
    if (bp2329_send_scpi (ID,buf) != 0)
    {
     bp2329_close_instr_gpib (ID);
     return bp2329_err;
    }
    
    /* Operation und Questionable register + SRQ Register freigeben*/
    Fmt(buf,"*SRE %i",(bp2329_SRQ_OPER_REG + bp2329_SRQ_QUEST_REG));
    if (bp2329_send_scpi (ID,buf) != 0)
    {
     bp2329_close_instr_gpib (ID);
     return bp2329_err;
    }

    *instrID = ID;

    
    return(0);
}
/*=========================================================================*/
/* This function opens a com port for the instrument module                */
/*=========================================================================*/
int _VI_FUNC bp2329_init_serial(int port,int baud_rate,double timeout,char* model,int reset,int *instrID)
{
    int ID;

    interface = bp2329_SERIAL;   /* Serial */

    bp2329_err = 0;
    bp2329_setup_arrays ();

    if (bp2329_invalid_integer_range (port, 1,8, -1) != 0)
     return bp2329_err;
    if (bp2329_invalid_integer_range ((int)baud_rate, 1200, 38400, -2) != 0)
     return bp2329_err;
    if (bp2329_invalid_real_range (timeout, 0.0F, 72000.0F, -3) != 0)
     return bp2329_err;
    if (bp2329_invalid_integer_range (reset, 0,1, -7) != 0)
     return bp2329_err;

    ID = bp2329_open_instr_serial(port,baud_rate,timeout);
    if(ID <= 0)
     return bp2329_err;


    if (reset == 1)
    {
     if (bp2329_send_scpi (ID, "*RST") != 0)
     {
      bp2329_close_instr_serial(ID);
      return bp2329_err;
     }
    }
    if (bp2329_receive_scpi(ID,"*IDN?",model) != 0)
    {
     bp2329_close_instr_serial (ID);
     return bp2329_err;
    }
    *instrID = ID;

    return(0);
}
/* ========================================================================= */
/*  This function puts the device into local mode.                              */
/* ========================================================================= */
int _VI_FUNC  bp2329_local_gpib(int instrID)
{
    bp2329_err = 0;
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
    return bp2329_err;
    if (bp2329_device_closed_gpib (instrID) != 0)
    return bp2329_err;
    if ((ibloc (bd[instrID]) & (int)0x8000 != 0) != 0)
    {
       bp2329_err = 234;
       return bp2329_err;
    }
    return bp2329_err;
}
/*=========================================================================*/
/* This function sets the config measurement parameters                    */
/*=========================================================================*/
int _VI_FUNC bp2329_set_config_meas(int instrID,int aver_ctrl,int aver_counts,int loading,
						int meas_mode,int limit_20mv,int meas_sequence,
						int num_of_cycles)
{
	bp2329_err = 0;
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
	
	if (bp2329_invalid_integer_range (aver_ctrl, 0,1, -2) != 0)
     return bp2329_err;
	if (bp2329_invalid_integer_range (aver_counts, 0,100, -3) != 0)
     return bp2329_err;
	if (bp2329_invalid_integer_range (loading, 0,1, -4) != 0)
     return bp2329_err;
	if (bp2329_invalid_integer_range (meas_mode, 0,1, -5) != 0)
     return bp2329_err;
	if (bp2329_invalid_integer_range (limit_20mv, 0,1, -6) != 0)
     return bp2329_err;
	if (bp2329_invalid_integer_range (meas_sequence, 0,3, -7) != 0)
     return bp2329_err;
	if (bp2329_invalid_integer_range (num_of_cycles, 0,3, -8) != 0)
     return bp2329_err;

	/*------------ Average Control--------------------*/
    if(bp2329_send_scpi(instrID,sense_aver_ctrl[aver_ctrl]) != 0)
     return bp2329_err;
	
	/*------------ Average Counts--------------------*/
	
    Fmt(buf,"%s<sens:aver:coun %d",aver_counts); 
    if(bp2329_send_scpi(instrID,buf) != 0)
     return bp2329_err;
	

	/*------------ meas Mode--------------------*/
    if(bp2329_send_scpi(instrID,sense_meas_mode[meas_mode]) != 0)
     return bp2329_err;
     
	/*------------ 20mV limit --------------------*/
    if(bp2329_send_scpi(instrID,sense_limit_20mv[limit_20mv]) != 0)
     return bp2329_err;

	/*------------ Loading--------------------*/
    if(bp2329_send_scpi(instrID,sense_loading[loading]) != 0)
     return bp2329_err;
     
	/*------------ meas sequence--------------------*/
	
    if(bp2329_send_scpi(instrID,sense_meas_sequence[meas_sequence]) != 0)
     return bp2329_err;
     
	/*------------ number of cycle--------------------*/
    if(bp2329_send_scpi(instrID,sense_num_of_cycles[num_of_cycles]) != 0)
     return bp2329_err;
	
	return bp2329_err;
	
}
/*=========================================================================*/
/* This function gets the config measurement parameters                    */
/*=========================================================================*/
int _VI_FUNC bp2329_get_config_meas(int instrID,int* aver_ctrl,int* aver_counts,int* loading,
						int* meas_mode,int* limit_20mv,int* meas_sequence,
						int* num_of_cycles)
{
	bp2329_err = 0;
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
	
	/*------------ Average Control--------------------*/
	Fmt(cmd,"%s<SENSE:AVER:TCON?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
     
	if(CompareStrings (buf, 0, "MOV", 0, 0) == 0)
      *aver_ctrl = 0;
    else
      *aver_ctrl = 1;
	
	/*------------ Average Counts--------------------*/
	
    Fmt(cmd,"%s<sens:aver:coun? "); 
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%d",aver_counts);
	
	/*------------ Loading--------------------*/
    Fmt(cmd,"%s<sens:fres:load?"); 
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
     
	if(CompareStrings (buf, 0, "REAL", 0, 0) == 0)
      *loading = 0;
    else
      *loading = 1;
     
	/*------------ meas Mode--------------------*/
    Fmt(cmd,"%s<init:cont?"); 
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%d",meas_mode);
     
	/*------------ 20mV limit --------------------*/
	Fmt(cmd,"%s<sour:volt:lim:stat?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%d",limit_20mv);
     
	/*------------ meas sequence--------------------*/
    Fmt(cmd,"%s<sens:fres:mode?"); 
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
     
	if(CompareStrings (buf, 0, "refc", 0, 0) == 0)
      *meas_sequence = 0;
	else if(CompareStrings (buf, 0, "nonc", 0, 0) == 0)
      *meas_sequence = 1;
	else if(CompareStrings (buf, 0, "onec", 0, 0) == 0)
      *meas_sequence = 2;
	else if(CompareStrings (buf, 0, "stan", 0, 0) == 0)
      *meas_sequence = 3;
    
     
	/*------------ number of cycle--------------------*/
    Fmt(cmd,"%s<sens:fres:nplc?"); 
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
     
	if(CompareStrings (buf, 0, "min", 0, 0) == 0)
      *num_of_cycles = 0;
	else if(CompareStrings (buf, 0, "med", 0, 0) == 0)
      *num_of_cycles = 1;
	else if(CompareStrings (buf, 0, "max", 0, 0) == 0)
      *num_of_cycles = 2;
	else if(CompareStrings (buf, 0, "stan", 0, 0) == 0)
      *num_of_cycles = 3;
	
	return bp2329_err;
}

/*=========================================================================*/
/* This function initialize the range                                      */
/*=========================================================================*/
int _VI_FUNC bp2329_set_config_range(int instrID,int range)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (range, 0, 7,  -2) != 0)
      return bp2329_err;
      
	if(range > 0)   /* No Auto Range => set the Autorange OFF*/
	{
     if(bp2329_send_scpi(instrID,"SENSE:FRES:RANG:AUTO OFF") != 0) /* Range OFF*/
      return bp2329_err;
	}
    if(bp2329_send_scpi(instrID,sense_range[range]) != 0) /* Range*/
     return bp2329_err;

    return(0);
}
/*=========================================================================*/
/* This function initialize the range                                      */
/*=========================================================================*/
int _VI_FUNC bp2329_get_config_range(int instrID,int* range)
{
	char range_txt[20];
	int auto_range;
	
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
    
      
    Fmt(cmd,"%s<SENSE:FRES:RANG:AUTO?"); 
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
     
	Scan(buf,"%s>%d,%s",&auto_range,range_txt);     
	
	if(auto_range == 1)
	{
		*range = 0;
	}
	else
	{
    	Fmt(cmd,"%s<SENSE:FRES:RANG:MAN?"); 
    	if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     		return bp2329_err;
     
		Scan(buf,"%s>%s",range_txt);     
		*range = get_range_number(range_txt);
	}	
	
    return(0);
}
/*=========================================================================*/
/* This function select the string to number                               */
/*=========================================================================*/

int get_range_number(char * range_txt)
{
	int range;
	
	if(CompareStrings (range_txt, 0, "200MOHM", 0, 0) == 0)
      range = 1;
    else  if(CompareStrings (range_txt, 0, "2OHM", 0, 0) == 0)
	  range = 2;
    else  if(CompareStrings (range_txt, 0, "20OHM", 0, 0) == 0)
	  range = 3;
    else  if(CompareStrings (range_txt, 0, "200OHM", 0, 0) == 0)
	  range = 4;
    else  if(CompareStrings (range_txt, 0, "2KOHM", 0, 0) == 0)
	  range = 5;
    else  if(CompareStrings (range_txt, 0, "20KOHM", 0, 0) == 0)
	  range = 6;
    else  if(CompareStrings (range_txt, 0, "200KOHM", 0, 0) == 0)
	  range = 7;
	
	return range;  
}
/*=========================================================================*/
/* This function config the auto range                                     */
/*=========================================================================*/
int _VI_FUNC bp2329_set_config_auto_range(int instrID,int lower_auto_range,int upper_auto_range)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (lower_auto_range, 1, 7,  -2) != 0)
      return bp2329_err;
      
    if (bp2329_invalid_integer_range (upper_auto_range, 1, 7,  -3) != 0)
      return bp2329_err;
    
    /* upper mu grer als lower Wert sein*/
    if(upper_auto_range  <= lower_auto_range)
    {
     bp2329_err = -4;    
     return bp2329_err;
	}
	
    if(bp2329_send_scpi(instrID,sense_lower_range[lower_auto_range]) != 0) /* lower auto Range*/
     return bp2329_err;
     
    if(bp2329_send_scpi(instrID,sense_upper_range[upper_auto_range]) != 0) /* upper auto Range*/
     return bp2329_err;                  
     
    return(0);
}
/*=========================================================================*/
/* This function get config the auto range                                     */
/*=========================================================================*/
int _VI_FUNC bp2329_get_config_auto_range(int instrID,int* lower_auto_range,int* upper_auto_range)
{
	char range_txt[20];
    bp2329_err = 0;
    

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;

      
    Fmt(cmd,"%s<SENSE:FRES:RANG:LOW?"); 
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
     
	Scan(buf,"%s>%s",range_txt);     
	*lower_auto_range = get_range_number(range_txt);
	
    Fmt(cmd,"%s<SENSE:FRES:RANG:UPP?"); 
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
     
	Scan(buf,"%s>%s",range_txt);     
	*upper_auto_range = get_range_number(range_txt);
	
    return(0);
}

/*=========================================================================*/
/* This function set the display mode                                      */
/*=========================================================================*/
int _VI_FUNC bp2329_set_config_display_mode(int instrID,int mode,double nom_value,int resolution)
{
    bp2329_err = 0;
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (mode, 0, 2,  -2) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (resolution, 0, 1 , -3) != 0)
      return bp2329_err;


  	Fmt(cmd,"%s<SENSE:FRES:RANG:MAN?"); 
  	if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
  		return bp2329_err;
    
    if(bp2329_send_scpi(instrID,display_mode[mode]) != 0)   /* display mode*/
     return bp2329_err;
     
	Fmt(buf,"%s<sens:fres:ref  %f",nom_value);         /* Nominal Value*/
    if(bp2329_send_scpi(instrID,buf) != 0)
     return bp2329_err;
     
    if(bp2329_send_scpi(instrID,display_resolution[resolution]) != 0)   /* display resolution*/
     return bp2329_err;
	
    return (0);
}
/*=========================================================================*/
/* This function get the display mode                                      */
/*=========================================================================*/
int _VI_FUNC bp2329_get_config_display_mode(int instrID,int *mode,double *nom_value,int* resolution)
{
	char value[30];
	
    bp2329_err = 0;
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;

    
	Fmt(cmd,"%s<calc:math:expr?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
     
	if(CompareStrings (buf, 0, "OHM", 0, 0) == 0)
      *mode = 0;
    else   if(CompareStrings (buf, 0, "DELTA", 0, 0) == 0)
  	  *mode = 1;
											
    else   if(CompareStrings (buf, 0, "LIMIT", 0, 0) == 0)
  	  *mode = 2;
  	  
	Fmt(cmd,"%s<sens:fres:ref?");         /* Nominal Value*/
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
    Scan(buf,"%s>%s",value);
    
    *nom_value = convert_string_to_value(value);
    
	Fmt(cmd,"%s<sens:fres:res?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
     
	if(CompareStrings (buf, 0, "0.0005", 0, 0) == 0)		/* 2000 DIGIT*/
      *resolution = 0;
    else 											/* 20000 DIGIT*/
  	  *resolution = 1;
	
    return (0);
}

/*=========================================================================*/
/* This function config the display options                                */
/*=========================================================================*/
int _VI_FUNC bp2329_set_config_display_options(int instrID,int range,int current,int temperature,
								  int sample,int trigger, int auto_man,int load,int measure)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (range, 0, 1,  -2) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (current, 0, 1,  -3) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (temperature, 0, 1,  -4) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (sample, 0, 1,  -5) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (trigger, 0, 1,  -6) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (auto_man, 0, 1,  -7) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (load, 0, 1,  -8) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (measure, 0, 1,  -9) != 0)
      return bp2329_err;
      
	Fmt(buf,"%s<disp:rang %d",range);
    if(bp2329_send_scpi(instrID,buf) != 0) /* Display range option*/
     return bp2329_err;
     
	Fmt(buf,"%s<disp:curr %d",current);
    if(bp2329_send_scpi(instrID,buf) != 0) /* Display current option*/
     return bp2329_err;
     
	Fmt(buf,"%s<disp:temp %d",temperature);
    if(bp2329_send_scpi(instrID,buf) != 0) /* Display Temperature option*/
     return bp2329_err;
     
	Fmt(buf,"%s<disp:samp %d",sample);
    if(bp2329_send_scpi(instrID,buf) != 0) /* Display Sample option*/
     return bp2329_err;
     
	Fmt(buf,"%s<disp:trig %d",trigger);
    if(bp2329_send_scpi(instrID,buf) != 0) /* Display Trigger option*/
     return bp2329_err;
     
	Fmt(buf,"%s<disp:auto %d",auto_man);
    if(bp2329_send_scpi(instrID,buf) != 0) /* Display Auto/Man option*/
     return bp2329_err;
     
	Fmt(buf,"%s<disp:load %d",load);
    if(bp2329_send_scpi(instrID,buf) != 0) /* Display Load Option*/
     return bp2329_err;
     
	Fmt(buf,"%s<disp:meas %d",measure);
    if(bp2329_send_scpi(instrID,buf) != 0) /* Display Measure option*/
     return bp2329_err;
     
    return(0);
}
int _VI_FUNC bp2329_get_config_display_options(int instrID,int* range,int *current,int* temperature,
								  int* sample,int* trigger, int* auto_man,int *load,int* measure)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
      
	Fmt(cmd,"%s<disp:rang?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
    Scan(buf,"%s>%d",range);			   /* Display range option*/
	
     
	Fmt(cmd,"%s<disp:curr?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
    Scan(buf,"%s>%d",current);			   /* Display current option*/  
	
     
	Fmt(cmd,"%s<disp:temp?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
    Scan(buf,"%s>%d",temperature);			   /* Display Temperature option*/  
	
     
	Fmt(cmd,"%s<disp:samp?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
    Scan(buf,"%s>%d",sample);			   /* Display Sample option*/  
	
  
	Fmt(cmd,"%s<disp:trig?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
    Scan(buf,"%s>%d",trigger);			   /* Display Trigger option*/  
	
	Fmt(cmd,"%s<disp:auto?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
    Scan(buf,"%s>%d",auto_man);			   /* Display Auto/Man option*/  
	
	Fmt(cmd,"%s<disp:load?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
    Scan(buf,"%s>%d",load);			   /* Display Load option*/  
	
     
	Fmt(cmd,"%s<disp:meas?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
    Scan(buf,"%s>%d",measure);			   /* Display Measure option*/  
	
     
    return(0);
}

/*=========================================================================*/
/* This function set the display contrast                                   */
/*=========================================================================*/
int _VI_FUNC bp2329_set_config_display_contrast(int instrID,double contrast)
{
    bp2329_err = 0;
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
    if (bp2329_invalid_real_range (contrast, 0.0, 1.0,  -2) != 0)
      return bp2329_err;


    Fmt(buf,"%s<:DISP:CONT %f",contrast);         /* Display contrast*/
    if(bp2329_send_scpi(instrID,buf) != 0)
     return bp2329_err;

    return (0);
}
/*=========================================================================*/
/* This function get the display contrast                                   */
/*=========================================================================*/
int _VI_FUNC bp2329_get_config_display_contrast(int instrID,double *contrast)
{
    bp2329_err = 0;
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;


    Fmt(cmd,"%s<:DISP:CONT?");         /* Display contrast*/
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
    Scan(buf,"%s>%f",contrast);			   /* Display Measure option*/  

    return (0);
}


/*=========================================================================*/
/* This function set the coefficient value                                 */
/*=========================================================================*/
int _VI_FUNC bp2329_set_config_temp_compensation(int instrID,int state,int type, double ref_value,double man_value)
{  
    bp2329_err = 0;
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (state, 0, 1,  -2) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (type, 0, 3, -3) != 0)
      return bp2329_err;
    if (bp2329_invalid_real_range (ref_value, -9999.9, 9999.9F,  -4) != 0)
      return bp2329_err;

    Fmt(buf,"%s<SENS:TCOM:STAT %d",state);
    if(bp2329_send_scpi(instrID,buf) != 0)/* set state ON/OFF*/
     return bp2329_err;


    if(bp2329_send_scpi(instrID,sense_tcomp[type]) != 0)/* set type */
     return bp2329_err;


     Fmt(buf,"%s<SENS:TCOM:TEMP:REF %f",ref_value);
     if(bp2329_send_scpi(instrID,buf) != 0)/* set Refernce Manual*/
      return bp2329_err;

    if(type == 0)    /* Manual */
    {
     if (bp2329_invalid_real_range (man_value, -9999.9, 9999.9F,  -5) != 0)
      return bp2329_err;

     Fmt(buf,"%s<SENS:TCOM:TEMP %f",man_value);
     if(bp2329_send_scpi(instrID,buf) != 0)/* set TK Manual*/
      return bp2329_err;
    }


    return(0);
}
/*=========================================================================*/
/* This function get the coefficient value                                 */
/*=========================================================================*/
int _VI_FUNC bp2329_get_config_temp_compensation(int instrID,int *state,int *type, double* ref_value, double* man_value)
{  
    bp2329_err = 0;
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;

    Fmt(cmd,"%s<SENS:TCOM:STAT?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)/* get state ON/OFF*/
     return bp2329_err;
	Scan(buf,"%s>%d",state);
	
    Fmt(cmd,"%s<SENS:TCOM?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	if(CompareStrings (buf, 0, "MAN", 0, 0) == 0)		/* Manuell*/
      *type = 0;
    else if(CompareStrings (buf, 0, "PT100", 0, 0) == 0)		/* PT100*/
  	  *type = 1;
    else if(CompareStrings (buf, 0, "UINP", 0, 0) == 0)			/* UINP*/
  	  *type = 2;
    else if(CompareStrings (buf, 0, "IINP", 0, 0) == 0)			/* IINP*/
  	  *type = 3;

   	Fmt(cmd,"%s<SENS:TCOM:TEMP:REF?");
  	if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
   		return bp2329_err;
	*ref_value = atof(buf);


    if(*type == 0)    /* Manual */
    {
     	Fmt(cmd,"%s<SENS:TCOM:TEMP?");
    	if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     		return bp2329_err;
		*man_value = atof(buf);
    }


    return(0);
}
/*=========================================================================*/
/* This function get the temp.value if automatic tempcomp is enabled       */
/*=========================================================================*/
int _VI_FUNC bp2329_get_temp_value(int instrID,double* value)
{  
    bp2329_err = 0;
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;

   	Fmt(cmd,"%s<SENS:TCOM:TEMP?");
  	if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
		return bp2329_err;
	*value = atof(buf);


    return(0);
}


/*=========================================================================*/
/* This function set config the PT 100 R0 and Coefficient  a,b                 */
/*=========================================================================*/
int _VI_FUNC bp2329_set_config_scale_pt100(int instrID,double R0, double a, double b)

{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;

    Fmt(buf,"%s<SCAL:PT100 %f,%f,%f",R0,a,b);
    if(bp2329_send_scpi(instrID,buf) != 0) /* set PT100 Coefficient*/
     return bp2329_err;

    return(0);
}
/*=========================================================================*/
/* This function get config the PT 100 R0 and Coefficient  a,b                 */
/*=========================================================================*/
int _VI_FUNC bp2329_get_config_scale_pt100(int instrID,double *R0, double* a, double* b)

{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;

    Fmt(cmd,"%s<SCAL:PT100?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0) /* get PT100 Coefficient*/
     return bp2329_err;
    Scan(buf,"%s>%f,%f,%f",R0,a,b);

    return(0);
}

/*=========================================================================*/
/* This function config sacle the Voltage or Current Input                 */
/*=========================================================================*/
int _VI_FUNC bp2329_set_config_scale_ui_input(int instrID,int mode, double value1, double value2,
								   double temp1, double temp2)

{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
      
    if (bp2329_invalid_integer_range (mode, 0, 1,  -2) != 0)
      return bp2329_err;
    if (bp2329_invalid_real_range (temp1, -200.0F,800.0F,  -5) != 0)
     return bp2329_err;
    if (bp2329_invalid_real_range (temp2, -200.0F,800.0F,  -6) != 0)
      return bp2329_err;

    if(mode == 0) /* U-Input*/
    {
     if (bp2329_invalid_real_range (value1, 0.0F,10.0F,  -3) != 0)
      return bp2329_err;
     if (bp2329_invalid_real_range (value2, 0.0F,10.0F,  -4) != 0)
      return bp2329_err;
    
     Fmt(buf,"%s<SCAL:VOLT %f,%f,%f,%f",value1,value2,temp1,temp2);
    } 
    else
    {
     if (bp2329_invalid_real_range (value1, 0.0F,20.0F,  -3) != 0)
      return bp2329_err;
     if (bp2329_invalid_real_range (value2, 0.0F,20.0F,  -4) != 0)
      return bp2329_err;
    
     Fmt(buf,"%s<SCAL:CURR %fMA,%fMA,%f,%f",value1,value2,temp1,temp2);
    } 
    if(bp2329_send_scpi(instrID,buf) != 0) /* set Linear values*/
     return bp2329_err;

    return(0);
}
/*=========================================================================*/
/* This function get config sacle the Voltage or Current Input                 */
/*=========================================================================*/
int _VI_FUNC bp2329_get_config_scale_ui_input(int instrID,int mode, double *value1, double *value2,
								   double *temp1, double *temp2)

{
	char help1[20],help2[20],help3[20],help4[20]; 
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
      
    if (bp2329_invalid_integer_range (mode, 0, 1,  -2) != 0)
      return bp2329_err;

    if(mode == 0) /* U-Input*/
     Fmt(cmd,"%s<SCAL:VOLT?");
    else
     Fmt(cmd,"%s<SCAL:CURR?");

    if(bp2329_receive_scpi(instrID,cmd,buf) != 0) /* get Linear values*/
     return bp2329_err;
	Scan(buf,"%s>%s[t44],%s[t44],%s[t44],%s[t44]",help1,help2,help3,help4);
	*value1 =  atof(help1);
	*value2 = atof(help2);
	*temp1 = atof(help3);
	*temp2 = atof(help4);
	
    return(0);
}


/*=========================================================================*/
/* This function set the coefficient(1-10) value                           */
/*=========================================================================*/
int _VI_FUNC bp2329_set_config_temp_coefficient(int instrID,int tk_number, int value)

{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (tk_number, 1, 10,  -2) != 0)
      return bp2329_err;

    Fmt(buf,"%s<SENS:TCOM:TCO %i,%i",tk_number,value);
    if(bp2329_send_scpi(instrID,buf) != 0)/* set TK*/
     return bp2329_err;

    return(0);
}
/*=========================================================================*/
/* This function get the coefficient(1-10) value                           */
/*=========================================================================*/
int _VI_FUNC bp2329_get_config_temp_coefficient(int instrID,int tk_number, int* value)

{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (tk_number, 1, 10,  -2) != 0)
      return bp2329_err;

    Fmt(cmd,"%s<SENS:TCOM:TCO? %i",tk_number);
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)/* get TK*/
     return bp2329_err;
	Scan(buf,"%s>%i",value);
    return(0);
}

/*=========================================================================*/
/* This function select the coefficient(1-10) value                           */
/*=========================================================================*/
int _VI_FUNC bp2329_set_active_temp_coefficient(int instrID,int tk_number)

{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (tk_number, 1, 10,  -2) != 0)
      return bp2329_err;

    Fmt(buf,"%s<SENS:TCOM:TCO:SEL %i",tk_number);
    if(bp2329_send_scpi(instrID,buf) != 0)/* set TK*/
     return bp2329_err;

    return(0);
}
/*=========================================================================*/
/* This function get the active the coefficient(1-10) value                           */
/*=========================================================================*/
int _VI_FUNC bp2329_get_active_temp_coefficient(int instrID,int *tk_value)

{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;

    Fmt(cmd,"%s<SENS:TCOM:TCO:SEL?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)/* get TK*/
     return bp2329_err;
	Scan(buf,"%s>%i",tk_value);
    return(0);
}

/*=========================================================================*/
/* This function set the deviec access for the menue                       */
/*=========================================================================*/
int _VI_FUNC bp2329_set_config_device_access(int instrID,int measure,int display,int printer,
								int temp_comp,int contrast,int store,int interface,
								int display_state,int in_out,int scale,int datalogger,
								int calibration,int date,int fast,int comp_conf,
								int comp_eval,int envelope_conf,
								int envelope_eval,int man_range)
{
    bp2329_err = 0;
    
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (measure, 0, 1,  -2) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (display, 0, 1,  -3) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (printer, 0, 1,  -4) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (temp_comp, 0, 1,  -5) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (interface, 0, 1,  -6) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (store, 0, 1,  -7) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (contrast, 0, 1,  -8) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (display_state, 0, 1,  -9) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (in_out, 0, 1,  -10) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (calibration, 0, 1,  -11) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (datalogger, 0, 1,  -12) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (scale, 0, 1,  -13) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (date, 0, 1,  -14) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (fast, 0, 1,  -15) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (comp_conf, 0, 1,  -16) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (comp_eval, 0, 1,  -17) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (envelope_conf, 0, 1,  -18) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (envelope_eval, 0, 1,  -19) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (man_range, 0, 1,  -20) != 0)
      return bp2329_err;
    
    Fmt(buf,"%s<acc:meas %d",measure);
    if(bp2329_send_scpi(instrID,buf) != 0) 
     return bp2329_err;
    Fmt(buf,"%s<acc:expr %d",display);
    if(bp2329_send_scpi(instrID,buf) != 0) 
     return bp2329_err;
    Fmt(buf,"%s<acc:prin %d",printer);
    if(bp2329_send_scpi(instrID,buf) != 0) 
     return bp2329_err;
    Fmt(buf,"%s<acc:comp %d",temp_comp);
    if(bp2329_send_scpi(instrID,buf) != 0) 
     return bp2329_err;
    Fmt(buf,"%s<acc:mem %d",store);
    if(bp2329_send_scpi(instrID,buf) != 0) 
     return bp2329_err;
    Fmt(buf,"%s<acc:comm %d",interface);
    if(bp2329_send_scpi(instrID,buf) != 0) 
     return bp2329_err;
    Fmt(buf,"%s<acc:cont %d",contrast);
    if(bp2329_send_scpi(instrID,buf) != 0) 
     return bp2329_err;
    Fmt(buf,"%s<acc:stat %d",display_state);
    if(bp2329_send_scpi(instrID,buf) != 0) 
     return bp2329_err;
    Fmt(buf,"%s<acc:reg %d",in_out);
    if(bp2329_send_scpi(instrID,buf) != 0) 
     return bp2329_err;
    Fmt(buf,"%s<acc:cal %d",calibration);
    if(bp2329_send_scpi(instrID,buf) != 0) 
     return bp2329_err;
    Fmt(buf,"%s<acc:dat %d",datalogger);
    if(bp2329_send_scpi(instrID,buf) != 0) 
     return bp2329_err;
    Fmt(buf,"%s<acc:scal %d",scale);
    if(bp2329_send_scpi(instrID,buf) != 0) 
     return bp2329_err;
    Fmt(buf,"%s<acc:date %d",date);
    if(bp2329_send_scpi(instrID,buf) != 0) 
     return bp2329_err;
    Fmt(buf,"%s<acc:fast %d",fast);
    if(bp2329_send_scpi(instrID,buf) != 0) 
     return bp2329_err;
    Fmt(buf,"%s<acc:lim:conf %d",comp_conf);
    if(bp2329_send_scpi(instrID,buf) != 0) 
     return bp2329_err;
    Fmt(buf,"%s<acc:lim:eval %d",comp_eval);
    if(bp2329_send_scpi(instrID,buf) != 0) 
     return bp2329_err;
    Fmt(buf,"%s<acc:env:conf %d",envelope_conf);
    if(bp2329_send_scpi(instrID,buf) != 0) 
     return bp2329_err;
    Fmt(buf,"%s<acc:env:eval %d",envelope_eval);
    if(bp2329_send_scpi(instrID,buf) != 0) 
     return bp2329_err;
    Fmt(buf,"%s<acc:manr %d",man_range);
    if(bp2329_send_scpi(instrID,buf) != 0) 
     return bp2329_err;
    
    return(0);
}
/*=========================================================================*/
/* This function get the deviec access for the menue                       */
/*=========================================================================*/
int _VI_FUNC bp2329_get_config_device_access(int instrID,int* measure,int* display,int* printer,
								int* temp_comp,int* contrast,int* store,int* interface,
								int* display_state,int* in_out,int* scale,int* datalogger,
								int* calibration,int* date,int* fast,int* comp_conf,
								int* comp_eval,int* envelope_conf,
								int* envelope_eval,int* man_range)
{
    bp2329_err = 0;
    
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
    
    Fmt(cmd,"%s<acc:meas?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",measure);
    
    Fmt(cmd,"%s<acc:expr?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",display);
    
    Fmt(cmd,"%s<acc:prin?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",printer);
    
    Fmt(cmd,"%s<acc:comp?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",temp_comp);
    
    Fmt(cmd,"%s<acc:mem?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",store);
    
    Fmt(cmd,"%s<acc:comm?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",interface);
    
    Fmt(cmd,"%s<acc:cont?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",contrast);
    
    Fmt(cmd,"%s<acc:stat?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",display_state);
    
    Fmt(cmd,"%s<acc:reg?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",in_out);
    
    Fmt(cmd,"%s<acc:cal?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",calibration);
    
    Fmt(cmd,"%s<acc:dat?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",datalogger);
    
    Fmt(cmd,"%s<acc:scal?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",scale);
    
    Fmt(cmd,"%s<acc:date?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",date);
    
    Fmt(cmd,"%s<acc:fast?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",fast);
    
    Fmt(cmd,"%s<acc:lim:conf?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",comp_conf);
    
    Fmt(cmd,"%s<acc:lim:eval?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",comp_eval);
    
    Fmt(cmd,"%s<acc:env:conf?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",envelope_conf);
    
    Fmt(cmd,"%s<acc:env:eval?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",envelope_eval);
    
    Fmt(cmd,"%s<acc:manr?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",man_range);
    
    return(0);
}

/*=========================================================================*/
/* This function start or stop the printer                                 */
/*=========================================================================*/
int _VI_FUNC bp2329_set_config_printer_state(int instrID,int state)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (state, 0, 1,  -2) != 0)
      return bp2329_err;

	if(state == 0)
     Fmt(buf,"%s<HCOP:ABORT");
    else
     Fmt(buf,"%s<HCOP");
     
    if(bp2329_send_scpi(instrID,buf) != 0)/* start or stop the printer*/
     return bp2329_err;


    return(0);
}

/*=========================================================================*/
/* This function set config the printer mode                                   */
/*=========================================================================*/
int _VI_FUNC bp2329_set_config_printer_mode(int instrID,int temp,int counter,int label,
							   int time, int date,int meas_header,
							   char* label_text,int hh, int mm,int ss)
{
    int pos,len;
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (temp, 0, 1,  -2) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (counter, 0, 1,  -3) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (label, 0, 1,  -4) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (time, 0, 1,  -5) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (date, 0, 1,  -6) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (meas_header, 1, 1000,  -7) != 0)
      return bp2329_err;
    if((len = strlen(label_text)) > 10)
    {
     bp2329_err = -8;
     return bp2329_err; 
    }
    if (bp2329_invalid_integer_range (hh, 0, 99,  -9) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (mm, 0, 59,  -10) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (ss, 0, 59,  -11) != 0)
      return bp2329_err;

	
    Fmt(buf,"%s<HCOP:ITEM:TEMP:STAT %d",temp);
    if(bp2329_send_scpi(instrID,buf) != 0)
     return bp2329_err;

    Fmt(buf,"%s<HCOP:ITEM:NUM:STAT %d",counter);
    if(bp2329_send_scpi(instrID,buf) != 0)
     return bp2329_err;

    Fmt(buf,"%s<HCOP:ITEM:LAB:STAT %d",label);
    if(bp2329_send_scpi(instrID,buf) != 0)
     return bp2329_err;

    Fmt(buf,"%s<HCOP:ITEM:TIME:STAT %d",time);
    if(bp2329_send_scpi(instrID,buf) != 0)
     return bp2329_err;
     
    Fmt(buf,"%s<HCOP:ITEM:DATE:STAT %d",date);
    if(bp2329_send_scpi(instrID,buf) != 0)
     return bp2329_err;
     
    Fmt(buf,"%s<HCOP:HEAD %d",meas_header);
    if(bp2329_send_scpi(instrID,buf) != 0)
     return bp2329_err;
     
    /* change Space in _ (underline)*/
	while((pos = FindPattern (label_text, 0, len, " ", 0, 0)) != -1)
	 label_text[pos] = '_';
     
    Fmt(buf,"%s<HCOP:ITEM:LAB:TEXT %s",label_text);
    if(bp2329_send_scpi(instrID,buf) != 0)
     return bp2329_err;

    Fmt(buf,"%s<HCOP:TDST %d[w2p0],%d[w2p0],%d[w2p0]",hh,mm,ss);
    if(bp2329_send_scpi(instrID,buf) != 0)
     return bp2329_err;
     

    return(0);
}
/*=========================================================================*/
/* This function get config the printer mode                                   */
/*=========================================================================*/
int _VI_FUNC bp2329_get_config_printer_mode(int instrID,int *temp,int* counter,int* label,
							   int *time, int *date,int *meas_header,
							   char* label_text,int* hh, int* mm,int* ss)
{
    int pos,len;
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;

	
    Fmt(cmd,"%s<HCOP:ITEM:TEMP:STAT?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",temp);
    
    Fmt(cmd,"%s<HCOP:ITEM:NUM:STAT?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",counter);
    

    Fmt(cmd,"%s<HCOP:ITEM:LAB:STAT?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",label);
    

    Fmt(cmd,"%s<HCOP:ITEM:TIME:STAT?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",time);
    
     
    Fmt(cmd,"%s<HCOP:ITEM:DATE:STAT?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",date);
    
     
    Fmt(cmd,"%s<HCOP:HEAD?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%i",meas_header);
    
     
    Fmt(cmd,"%s<HCOP:ITEM:LAB:TEXT?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%s",label_text);
	
    Fmt(cmd,"%s<HCOP:TDST?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%d,%d,%d",hh,mm,ss);
     

    return(0);
}

/*=========================================================================*/
/* This function Reset the counter of printer                              */
/*=========================================================================*/
int _VI_FUNC bp2329_printer_counter_reset(int instrID)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;

    Fmt(buf,"%s<HCOP:NUM:RES");
    if(bp2329_send_scpi(instrID,buf) != 0)
     return bp2329_err;

    return(0);
}
/* ========================================================================= */
/*  This function recall memory label                                         */
/* ========================================================================= */
int _VI_FUNC bp2329_get_config_memory_label(int instrID,int mem_no,char* label)
{
    int ret;

    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
    if (bp2329_invalid_integer_range (mem_no, 0, 32,  -2) != 0)
     return bp2329_err;

    Fmt (cmd, "%s<mem:stat:name? %d",mem_no);
    if (bp2329_receive_scpi (instrID, cmd,buf) != 0)
      return bp2329_err;
	Scan(buf,"%s>%s",label);
    return (0);
}
/* ========================================================================= */
/*  This function store memory label                                         */
/* ========================================================================= */
int _VI_FUNC bp2329_set_config_memory_label(int instrID,int mem_no,char* label)
{
    int ret,pos,len;

    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
    if (bp2329_invalid_integer_range (mem_no, 0, 32,  -2) != 0)
     return bp2329_err;
    if((len = strlen(label)) > 10)
    {
     bp2329_err = -3;
     return bp2329_err; 
    }

    /* change Space in _ (underline)*/
	while((pos = FindPattern (label, 0, len-1, " ", 0, 0)) != -1)
	 label[pos] = '_';

    Fmt (cmd, "%s<mem:stat:def %s,%d",label,mem_no);
    if (bp2329_send_scpi (instrID, cmd) != 0)
      return bp2329_err;

    return (0);
}

/* ========================================================================= */
/*  This function store memory no                                            */
/* ========================================================================= */
int _VI_FUNC bp2329_set_config_memory(int instrID,int mem_no)
{
    int ret;

    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
    if (bp2329_invalid_integer_range (mem_no, 0, 32,  -2) != 0)
     return bp2329_err;

    Fmt (cmd, "%s<*SAV %i",mem_no);
    if (bp2329_send_scpi (instrID, cmd) != 0)
      return bp2329_err;

    return (0);
}
/* ========================================================================= */
/*  This function recall memory no                                            */
/* ========================================================================= */
int _VI_FUNC bp2329_get_config_memory(int instrID,int mem_no)
{
    int ret;

    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
    if (bp2329_invalid_integer_range (mem_no, 0, 32,  -2) != 0)
     return bp2329_err;

    Fmt (cmd, "%s<*RCL %i",mem_no);
    if (bp2329_send_scpi (instrID, cmd) != 0)
      return bp2329_err;

    return (0);
}

/*=========================================================================*/
/* This function start the measurnment                                     */
/*=========================================================================*/
int _VI_FUNC bp2329_meas_start(int instrID)
{
	int status; 
	
    bp2329_err = 0;
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
    
    if(interface == bp2329_GPIB)  /* GPIB ?? */
    {
    /* if(bp2329_receive_scpi(instrID,":STAT:OPER:COND?",buf) != 0)
      return (bp2329_err);
     Scan(buf,"%s>%i",&status);
     if(status & bp2329_MEASURING)
      return -1;*/
	}

	if(bp2329_send_scpi(instrID,"IN") != 0)
     return bp2329_err;

    return (0);

}
/*=========================================================================*/
/* This function does something                                            */
/*=========================================================================*/
int _VI_FUNC  bp2329_spoll(int instrID,unsigned char* srq)
{
 
    ibrsp(bd[instrID], srq);
    if( ibsta & ERR )
    {
       bp2329_err = 304;
       return bp2329_err;
    }
    return 0;
}
/*=========================================================================*/
/* This function read the measurnment value with unit                      */
/*=========================================================================*/
int _VI_FUNC bp2329_meas_read(int instrID,double* value ,char* unit)
{
    int reg_status,device_status,meas_status,ret;
    int done ,mask;
    unsigned char srq ;

    bp2329_err = 0;
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;


 /*   if(bp2329_receive_scpi(instrID,"FE?",buf) != 0)
    return bp2329_err;
    
    Scan(buf,"%s>%f%s",value,unit);			
    return (0);										/*+++++++++++++*/

    if(interface == bp2329_GPIB)  /* GPIB ?? */
    {
     /* IM GPIB-Betrieb darf der Messwert nicht direkt abgefragt werden,
      => Abfrage ber SRQ. Ansonsten wird der 2329 zugefahren*/
     meas_status = 0;
     while(!(meas_status & bp2329_EOC))
     {
     
      done = 0;
      mask = TIMO|RQS;
      srq = 0;
      while(done != 1)
      {
       
       ibwait (bd[instrID], mask) ;
       if( ibsta & (ERR|TIMO) )
       {
        bp2329_err = 303;
        return bp2329_err;
       }

       ibrsp(bd[instrID], &srq);   /* Spoll Byte abholen*/
       if( ibsta & ERR )
       {
         bp2329_err = 304;
         return bp2329_err;
       }


    /*   FmtOut ("\n Srq  1 = %d",srq);*/

	   switch(srq)
	   {
	    default:
	     break;
	    case 192:   /* RQS + Operation Status Register*/
         done = 1;
         ret = bp2329_read_oper_event_reg(instrID,&meas_status);
         if(ret != 0)
          return (ret);
         
         break;
        case 72:   /* RQS + Questionable Register*/
        case 200:   /* RQS + Questionable Register+Operation Register*/
         done = 1;
          
         /* -- das Operation und Questionable register mu ausgelsen werden
               damit sie auch zurckgesetzt werden --------------------------*/
         ret = bp2329_read_oper_event_reg(instrID,&reg_status);
         if(ret != 0)
          return (ret);
	
         ret = bp2329_read_quest_reg(instrID,&reg_status);
         if(ret != 0)
          return (ret);
          
         /* ------------------- read FRes Status -----------------*/
         ret = bp2329_read_quest_fres_reg(instrID,&device_status);
         if(ret != 0)
         {
          *value = 0.0;
          return (ret);
         }
         
         return (0);
         
         break;
        case 0:   /* no Value in Srq - Byte*/
         /*FmtOut("\nsrq %i meas_status %i",srq,meas_status);*/
         if(!(meas_status & bp2329_MEASURING))    /* no Meas Flag*/
          return 302;
         break;
       }
      }
     } 
     /*------------------ Den Messwert des 2329 abfragen--------------- */

     if(bp2329_receive_scpi(instrID,"FE?",buf) != 0)
      return bp2329_err;

     Scan(buf,"%s>%f%s",value,unit);
    }
    else
    {
     /*------------------ Den Messwert des 2329 abfragen--------------- */
     ret = bp2329_read_oper_event_reg(instrID,&meas_status);
     if(ret != 0)
      return (ret);
     if(!(meas_status & bp2329_EOC))    /* no EOC Flag*/
      return 301;

     ret = bp2329_receive_scpi(instrID,"FE?",buf);
     switch (ret)
     {
      case 0:					/* all OK*/
	   if(CompareStrings (buf, 0, ">>>", 0, 0) == 0)	/* Device ERROR*/
       {
        /* ------------------- read FRes Status -----------------*/
        ret = bp2329_read_quest_fres_reg(instrID,&device_status);
        if(ret != 0)
        {
         *value = 0.0;
         return (ret);
        }
        return (0);
       }
       else
        Scan(buf,"%s>%f%s",value,unit);
       break;
      case 301:					/* EOT >= Measure Value */
       break; 
      default:
       break;
     }
    }

    return (0);
}
/*=========================================================================*/
/* This function read the measurnment value with unit                      */
/*=========================================================================*/
int _VI_FUNC bp2329_meas_read_string(int instrID,char* value )
{
    int reg_status,device_status,meas_status,ret;
    int done ,mask;
    unsigned char srq ;
    

    bp2329_err = 0;
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;


 /*   if(bp2329_receive_scpi(instrID,"FE?",buf) != 0)
    return bp2329_err;
    
    Scan(buf,"%s>%f%s",value,unit);			
    return (0);										/*+++++++++++++*/

    if(interface == bp2329_GPIB)  /* GPIB ?? */
    {
     /* IM GPIB-Betrieb darf der Messwert nicht direkt abgefragt werden,
      => Abfrage ber SRQ. Ansonsten wird der 2329 zugefahren*/
     meas_status = 0;
     while(!(meas_status & bp2329_EOC))
     {
     
      done = 0;
      mask = TIMO|RQS;
      srq = 0;
      while(done != 1)
      {
       
       ibwait (bd[instrID], mask) ;
       if( ibsta & (ERR|TIMO) )
       {
        bp2329_err = 303;
        return bp2329_err;
       }

       ibrsp(bd[instrID], &srq);   /* Spoll Byte abholen*/
       if( ibsta & ERR )
       {
         bp2329_err = 304;
         return bp2329_err;
       }


    /*   FmtOut ("\n Srq  1 = %d",srq);*/

	   switch(srq)
	   {
	    default:
	     break;
	    case 192:   /* RQS + Operation Status Register*/
         done = 1;
         ret = bp2329_read_oper_event_reg(instrID,&meas_status);
         if(ret != 0)
          return (ret);
         
         break;
        case 72:   /* RQS + Questionable Register*/
        case 200:   /* RQS + Questionable Register+Operation Register*/
         done = 1;
          
         /* -- das Operation und Questionable register mu ausgelsen werden
               damit sie auch zurckgesetzt werden --------------------------*/
         ret = bp2329_read_oper_event_reg(instrID,&reg_status);
         if(ret != 0)
          return (ret);
	
         ret = bp2329_read_quest_reg(instrID,&reg_status);
         if(ret != 0)
          return (ret);
          
         /* ------------------- read FRes Status -----------------*/
         ret = bp2329_read_quest_fres_reg(instrID,&device_status);
         if(ret != 0)
         {
          value = 0x00;
          return (ret);
         }
         
         return (0);
         
         break;
        case 0:   /* no Value in Srq - Byte*/
         /*FmtOut("\nsrq %i meas_status %i",srq,meas_status);*/
         if(!(meas_status & bp2329_MEASURING))    /* no Meas Flag*/
          return 302;
         break;
       }
      }
     } 
     /*------------------ Den Messwert des 2329 abfragen--------------- */

     if(bp2329_receive_scpi(instrID,"FE?",buf) != 0)
      return bp2329_err;

     strcpy(value,buf);
     
    }
    else		/* Serial*/
    {
     ret = bp2329_read_oper_event_reg(instrID,&meas_status);
     if(ret != 0)
      return (ret);
     if(!(meas_status & bp2329_EOC))    /* no EOC Flag*/
      return 301;
    
     ret = bp2329_receive_scpi(instrID,"FE?",buf);
     switch (ret)
     {
      case 309:
       ret = 0;
      case 0:					/* all OK*/
	   if(FindPattern (buf, 0, 30, ">>>", 0, 0) >= 0)	/* Device ERROR*/
       {
        /* ------------------- read FRes Status -----------------*/
        ret = bp2329_read_quest_fres_reg(instrID,&device_status);
        if(ret != 0)
        {
         value = 0x00;
         return (ret);
        }
        return (0);
       }
       else
        strcpy(value,buf);
       break;
      case 301:					/* EOT >= Measure Value */
       break;
     }
    }

    return (0);
}


/*=========================================================================*/
/* This function stop the measurnment                                     */
/*=========================================================================*/
int _VI_FUNC  bp2329_meas_stop(int instrID)
{
    unsigned char srq ;
    int reg_status,ret;
    
    bp2329_err = 0;
    
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
    if(bp2329_send_scpi(instrID,"AB") != 0)
     return bp2329_err;
     
    

    if(interface == bp2329_GPIB)  /* GPIB ?? */
    {
     /* -- das Operation und Questionable register mu ausgelesen werden
          damit sie auch zurckgesetzt werden --------------------------*/
     ret = bp2329_read_oper_event_reg(instrID,&reg_status);
     if(ret != 0)
      return (ret);

     ret = bp2329_read_quest_reg(instrID,&reg_status);
     if(ret != 0)
      return (ret);
          
     /*-------- die SRQ-Leitung wieder freigegeben -------- */
     if (ibrsp(bd[instrID], &srq) <= 0)
     {
       bp2329_err = iberr;
       return bp2329_err;
     }
     
     
     
	}
    return (0);
}

/* ========================================================================= */
/*  This function switch the envelope acquistition state ON/OFF              */
/* ========================================================================= */
int _VI_FUNC  bp2329_set_envelope_state(int instrID,int state)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
    if (bp2329_invalid_integer_range (state, 0, 1,  -2) != 0)
     return bp2329_err;

    Fmt (buf, "%s<SENS:ENV:STAT %i",state);
    if (bp2329_send_scpi (instrID, buf) != 0)
     return bp2329_err;

    return (0);
}
/* ========================================================================= */
/*  This function get the envelope acquistition state ON/OFF              */
/* ========================================================================= */
int _VI_FUNC  bp2329_get_envelope_state(int instrID,int *state)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;

    Fmt (cmd, "%s<SENS:ENV:STAT?");
    if (bp2329_receive_scpi (instrID,cmd, buf) != 0)
     return bp2329_err;
    Scan(buf,"%s>%d",state); 

    return (0);
}

/* ========================================================================= */
/*  This function read the extremities datas                                 */
/* ========================================================================= */
int _VI_FUNC  bp2329_envelope_read_report(int instrID,char* min_value,char* max_value,
                                           char* diff_value)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;

    if(bp2329_receive_scpi(instrID,"SENS:ENV:REP:DATA?",buf) != 0)
      return bp2329_err;

    Scan(buf,"%s>%s[t44],%s[t44],%s[t0]",min_value,max_value,diff_value);

    return (0);
}
/* ========================================================================= */
/*  This function clear the envelope datas                                   */
/* ========================================================================= */
int _VI_FUNC  bp2329_envelope_clear(int instrID)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
     
    Fmt (buf,"%s<SENS:ENV:CLE");
    if (bp2329_send_scpi (instrID, buf) != 0)
     return bp2329_err;

    return (0);
}
/* ========================================================================= */
/*  This function switch the comparator state ON/OFF                        */
/* ========================================================================= */
int _VI_FUNC bp2329_set_config_comparator_mode(int instrID,int state,int relais_mode, int on_error)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
    if (bp2329_invalid_integer_range (state, 0, 1,  -2) != 0)
     return bp2329_err;
    if (bp2329_invalid_integer_range (relais_mode, 0, 1,  -3) != 0)
     return bp2329_err;
    if (bp2329_invalid_integer_range (on_error, 0, 1,  -4) != 0)
     return bp2329_err;

    Fmt (buf, "%s<CALC:LIM:STAT %i",state);
    if (bp2329_send_scpi (instrID, buf) != 0)
     return bp2329_err;
     
    Fmt (buf, "%s<CALC:LIM:REL %i",relais_mode);
    if (bp2329_send_scpi (instrID, buf) != 0)
     return bp2329_err;
     
    if (bp2329_send_scpi (instrID, calc_limit_fault[on_error]) != 0)
     return bp2329_err;

    return (0);
}
/* ========================================================================= */
/*  This function switch the comparator state ON/OFF                        */
/* ========================================================================= */
int _VI_FUNC bp2329_get_config_comparator_mode(int instrID,int* state,int* relais_mode, int* on_error)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;

    Fmt (cmd, "%s<CALC:LIM:STAT?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
    Scan(buf,"%s>%d",state);	
    
     
    Fmt (cmd, "%s<CALC:LIM:REL?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
    Scan(buf,"%s>%d",relais_mode);	
    
    Fmt (cmd, "%s<calc:lim:faul?");
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
     return bp2329_err;
     
	if(CompareStrings (buf, 0, "NONE", 0, 0) == 0)	
      *on_error = 0;
    else 				
  	  *on_error = 1;

    return (0);
}


/* ========================================================================= */
/*  This function config the two limit comparator                            */
/* ========================================================================= */
int _VI_FUNC bp2329_set_config_2_limit_comparator(int instrID,double lower,double upper)
{
    char cbuf[300];
    int flag;

    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;

    Fmt (buf, "%s<CALC:LIM:COUN 2"); /* two Limit Comparator*/
    if (bp2329_send_scpi (instrID, buf) != 0)
     return bp2329_err;

    Fmt (buf, "%s<CALC:LIM:LOW %f",lower);
    if (bp2329_send_scpi (instrID, buf) != 0)
     return bp2329_err;

    Fmt (buf, "%s<CALC:LIM:UPP %f",upper);
    if (bp2329_send_scpi (instrID, buf) != 0)
     return bp2329_err;

    if(bp2329_receive_scpi(instrID,"CALC:LIM:ACKNOWLEDGE?",cbuf) != 0)/* Flag lesen */
     return bp2329_err;

    Scan(cbuf,"%s>%i",&flag);
    if(flag == 0)    /* No Correct Comparator values*/
     return (330);

    return (0);
}
/* ========================================================================= */
/*  This function get config the two limit comparator                            */
/* ========================================================================= */
int _VI_FUNC bp2329_get_config_2_limit_comparator(int instrID,double *lower,double *upper)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;


    Fmt (cmd, "%s<CALC:LIM:LOW?");
    if (bp2329_receive_scpi (instrID, cmd,buf) != 0)
     return bp2329_err;
    *lower = convert_string_to_value(buf);
    
    Fmt (cmd, "%s<CALC:LIM:UPP?");
    if (bp2329_receive_scpi (instrID, cmd,buf) != 0)
     return bp2329_err;
    *upper = convert_string_to_value(buf);


    return (0);
}

/* ========================================================================= */
/*  This function set config the four limit comparator                            */
/* ========================================================================= */
int _VI_FUNC bp2329_set_config_4_limit_comparator(int instrID,double limit1,double limit2,
                                       double limit3,double limit4)
{
    char cbuf[300];
    int flag;

    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;

    Fmt (buf, "%s<CALC:LIM:COUN 4"); /* Four Limit Comparator*/
    if (bp2329_send_scpi (instrID, buf) != 0)
     return bp2329_err;

    Fmt (buf, "%s<CALC:LIM:GW1 %f",limit1);
    if (bp2329_send_scpi (instrID, buf) != 0)
     return bp2329_err;

    Fmt (buf, "%s<CALC:LIM:GW2 %f",limit2);
    if (bp2329_send_scpi (instrID, buf) != 0)
     return bp2329_err;

    Fmt (buf, "%s<CALC:LIM:GW3 %f",limit3);
    if (bp2329_send_scpi (instrID, buf) != 0)
     return bp2329_err;

    Fmt (buf, "%s<CALC:LIM:GW4 %f",limit4);
    if (bp2329_send_scpi (instrID, buf) != 0)
     return bp2329_err;

    if(bp2329_receive_scpi(instrID,"CALC:LIM:ACKNOWLEDGE?",cbuf) != 0)/* Flag lesen */
     return bp2329_err;

    Scan(cbuf,"%s>%i",&flag);
    if(flag == 0)    /* No Correct Comparator values*/
     return (330);

    return (0);
}
/* ========================================================================= */
/*  This function get config the four limit comparator                            */
/* ========================================================================= */
int _VI_FUNC bp2329_get_config_4_limit_comparator(int instrID,double* limit1,double* limit2,
                                       double *limit3,double *limit4)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;


    Fmt (cmd, "%s<CALC:LIM:GW1?");
    if (bp2329_receive_scpi (instrID, cmd,buf) != 0)
     return bp2329_err;
    *limit1 = convert_string_to_value(buf);
    
    Fmt (cmd, "%s<CALC:LIM:GW2?");
    if (bp2329_receive_scpi (instrID, cmd,buf) != 0)
     return bp2329_err;
    *limit2 = convert_string_to_value(buf);
    
    Fmt (cmd, "%s<CALC:LIM:GW3?");
    if (bp2329_receive_scpi (instrID, cmd,buf) != 0)
     return bp2329_err;
    *limit3 = convert_string_to_value(buf);
    
    Fmt (cmd, "%s<CALC:LIM:GW4?");
    if (bp2329_receive_scpi (instrID, cmd,buf) != 0)
     return bp2329_err;
    *limit4 = convert_string_to_value(buf);


    return (0);
}

/* ========================================================================= */
/*  This function read the two limit comparartor datas                      */
/* ========================================================================= */
int _VI_FUNC  bp2329_comparator_2_limit_report(int instrID,int* value1,int* value2,
                                           int* value3)
{
    char dmy[3][10];
    
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;

    if(bp2329_receive_scpi(instrID,"CALC:LIM:REP?",buf) != 0)
      return bp2329_err;

    Scan(buf,"%s>%s%d,%s%d,%s%d",dmy[0],value1,dmy[1],value2,dmy[2],value3);

    return (0);
}
/* ========================================================================= */
/*  This function read the four limit comparartor datas                      */
/* ========================================================================= */
int _VI_FUNC  bp2329_comparator_4_limit_report(int instrID,int* value1,int* value2,
                                           int* value3,int* value4, int* value5)
{
    char dmy[5][10];
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;

    if(bp2329_receive_scpi(instrID,"CALC:LIM:REP?",buf) != 0)
      return bp2329_err;

    Scan(buf,"%s>%s%d,%s%d,%s%d,%s%d,%s%d",dmy[0],value1,dmy[1],value2,dmy[2],value3,dmy[3],
    									value4,dmy[4],value5);

    return (0);
}

/* ========================================================================= */
/*  This function clear the comparator datas                                 */
/* ========================================================================= */
int _VI_FUNC  bp2329_comparator_clear(int instrID)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
     
    Fmt (buf,"%s<CALC:LIM:CLE");
    if (bp2329_send_scpi (instrID, buf) != 0)
     return bp2329_err;

    return (0);
}
/*=========================================================================*/
/* This function switches the logg state on/off                            */
/*=========================================================================*/
int _VI_FUNC  bp2329_set_datalogger_state (int instrID, int state)
{
    bp2329_err = 0;
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
    if (bp2329_invalid_integer_range (state, 0, 1,  -2) != 0)
     return bp2329_err;
    

    Fmt (cmd, "%s<DAT:STAT %i",state);
    if (bp2329_send_scpi (instrID, cmd) != 0)
     return bp2329_err;

    return (0);

 }
/*=========================================================================*/
/* This function switches the logg state on/off                            */
/*=========================================================================*/
int _VI_FUNC  bp2329_get_datalogger_state (int instrID, int *state)
{
    bp2329_err = 0;
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
     
    Fmt (cmd, "%s<DAT:STAT?");
    if (bp2329_receive_scpi (instrID, cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%d",state);
    return (0);

 }
/*=========================================================================*/
/* This function set the datalogger  block name                            */
/*=========================================================================*/
int _VI_FUNC  bp2329_set_datalogger_name(int instrID, int block, char* name)
{
    int len,pos;
    
    bp2329_err = 0;
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
    if (bp2329_invalid_integer_range (block, 0, 31,  -2) != 0)
      return bp2329_err;
      
    if((len = strlen(name)) > 10)
    {
     bp2329_err = -4;
     return bp2329_err; 
    }
 
     
	while((pos = FindPattern (name, 0, len-1, " ", 0, 0)) != -1)
	 name[pos] = '_';   /* space to "_"*/

    Fmt (cmd, "%s<DAT:STATE:DEF %s,%i",name,block);
    if (bp2329_send_scpi (instrID, cmd) != 0)
     return bp2329_err;
     
    return (0);    
}
/*=========================================================================*/
/* This function get the datalogger  block name                     */
/*=========================================================================*/
int _VI_FUNC  bp2329_get_datalogger_name(int instrID, int block, char* name)
{
    int len,pos;
    
    bp2329_err = 0;
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
    if (bp2329_invalid_integer_range (block, 0, 31,  -2) != 0)
      return bp2329_err;
      

    Fmt (cmd, "%s<DAT:STATE:NAME? %i",block);
    if (bp2329_receive_scpi (instrID, cmd,buf) != 0)
     return bp2329_err;
    Scan(buf,"%s>%s",name); 
     
    return (0);    
}
/*=========================================================================*/
/* This function set config the datalogger  block                              */
/*=========================================================================*/
int _VI_FUNC  bp2329_set_datalogger_block_size(int instrID, int block,int size)
{
    int len,pos;
    
    bp2329_err = 0;
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
    if (bp2329_invalid_integer_range (block, 0, 31,  -2) != 0)
      return bp2329_err;
    
    if (bp2329_invalid_integer_range (size, 0, 20000,  -3) != 0)
      return bp2329_err;

    Fmt (cmd, "%s<DAT:SIZE %i,%i",block,size);
    if (bp2329_send_scpi (instrID, cmd) != 0)
     return bp2329_err;
     
   return 0;
}
/*=========================================================================*/
/* This function set config the datalogger  block                              */
/*=========================================================================*/
int _VI_FUNC  bp2329_get_datalogger_block_size(int instrID, int block, int *size)
{
    char filter_txt[20];
    
    bp2329_err = 0;
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
    if (bp2329_invalid_integer_range (block, 0, 31,  -2) != 0)
      return bp2329_err;
    

    Fmt (cmd, "%s<DAT:SIZE? %i",block);
    if (bp2329_receive_scpi (instrID, cmd,buf) != 0)
     return bp2329_err;
    Scan(buf,"%s>%d",size);
    
    return 0;
}
/*=========================================================================*/
/* This function set config the datalogger  block                              */
/*=========================================================================*/
int _VI_FUNC  bp2329_set_datalogger_filter(int instrID, int block,
                                int filter, int xval, int hh,int mm,int ss,
                                double delta)
{
    int len,pos;
    
    bp2329_err = 0;
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
    if (bp2329_invalid_integer_range (block, 0, 31,  -2) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (filter, 0, 5,  -3) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (xval, 2, 9999,  -4) != 0)
      return bp2329_err;
     



	switch(filter)
	{
	  case 0:
	   Fmt (cmd, "%s<DAT:FILT %i,%s",block,"ALL");
       if (bp2329_send_scpi (instrID, cmd) != 0)
        return bp2329_err;
	   
	   break;
	  case 1:
	   Fmt (cmd, "%s<DAT:FILT %i,%s",block,"NOFAIL");
       if (bp2329_send_scpi (instrID, cmd) != 0)
        return bp2329_err;
	   
	   break;
	  case 2:
	   Fmt (cmd, "%s<DAT:FILT %i,%s",block,"FAIL");
       if (bp2329_send_scpi (instrID, cmd) != 0)
        return bp2329_err;
	   
	   break;
	  case 3:
	   Fmt (cmd, "%s<DAT:FILT %i,%s",block,"XVAL");
       if (bp2329_send_scpi (instrID, cmd) != 0)
        return bp2329_err;
	   
       Fmt (cmd, "%s<DAT:FILT:XVAL %i,%i",block,xval);
       if (bp2329_send_scpi (instrID, cmd) != 0)
        return bp2329_err;
        
	   
	   break;
	  case 4:
       if (bp2329_invalid_integer_range (hh, 0, 99,  -7) != 0)
        return bp2329_err;
       if (bp2329_invalid_integer_range (mm, 0, 59,  -8) != 0)
        return bp2329_err;
       if (bp2329_invalid_integer_range (ss, 0, 59,  -9) != 0)
        return bp2329_err;
        
	   Fmt (cmd, "%s<DAT:FILT %i,%s",block,"YTIME");
       if (bp2329_send_scpi (instrID, cmd) != 0)
        return bp2329_err;
        
       Fmt(cmd,"%s<DAT:FILT:YTIM %d,%d[w2p0],%d[w2p0],%d[w2p0]",block,hh,mm,ss);
       if(bp2329_send_scpi(instrID,cmd) != 0)
        return bp2329_err;
	   
	   break;
	 case 5:
       if (bp2329_invalid_real_range (delta, 0.00001, 200000.0,  -10) != 0)
        return bp2329_err;
        
	   Fmt (cmd, "%s<DAT:FILT %i,%s",block,"DELTA");
       if (bp2329_send_scpi (instrID, cmd) != 0)
        return bp2329_err;
        
        
       Fmt (cmd, "%s<DAT:FILT:DELT %i,%f",block,delta);
       if (bp2329_send_scpi (instrID, cmd) != 0)
        return bp2329_err;
	   
	 break;
	}

	return (0);
}

/*=========================================================================*/
/* This function set config the datalogger  block                              */
/*=========================================================================*/
int _VI_FUNC  bp2329_get_datalogger_filter(int instrID, int block, 
                                int *filter, int* xval, int* hh,int* mm,int* ss,
                                double *delta)
{
    char filter_txt[20];
    
    bp2329_err = 0;
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
    if (bp2329_invalid_integer_range (block, 0, 31,  -2) != 0)
      return bp2329_err;
    

    
    Fmt (cmd, "%s<DAT:FILT? %i",block);
    if (bp2329_receive_scpi (instrID, cmd,buf) != 0)
     return bp2329_err;
    Scan(buf,"%s>%s",filter_txt);

	if 		(!CompareStrings (filter_txt, 0, "ALL", 0, 0))  *filter=0;
	else if (!CompareStrings (filter_txt, 0, "NOF", 0, 0))  *filter=1;
   	else if (!CompareStrings (filter_txt, 0, "FAIL", 0, 0)) *filter=2;
	else if (!CompareStrings (filter_txt, 0, "XVAL", 0, 0)) *filter=3;
	else if (!CompareStrings (filter_txt, 0, "YTIM", 0, 0)) *filter=4;
	else if (!CompareStrings (filter_txt, 0, "DELT", 0, 0)) *filter=5;
	
	Fmt(cmd,"%s<DAT:FILT:XVAL? %d",block);
    if (bp2329_receive_scpi (instrID, cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%d",xval);
		
	Fmt(cmd,"%s<DAT:FILT:YTIM? %d",block);
    if (bp2329_receive_scpi (instrID, cmd,buf) != 0)
     return bp2329_err;						 
    Scan(buf,"%s>%d,%d,%d",hh,mm,ss);
			
	Fmt(cmd,"%s<DAT:FILT:DELT? %d",block);
    if (bp2329_receive_scpi (instrID, cmd,buf) != 0)
     return bp2329_err;
	Scan(buf,"%s>%f",delta);
	

	return (0);
}


/*=========================================================================*/
/* This function clears a block                                            */
/*=========================================================================*/
int _VI_FUNC  bp2329_datalogger_clear_block (int instrID, int block)
{
     bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
    if (bp2329_invalid_integer_range (block, 0, 31,  -2) != 0)
      return bp2329_err;
   
    Fmt (cmd, "%s<DAT:CLEAR %i",block);
    if (bp2329_send_scpi (instrID, cmd) != 0)
     return bp2329_err;

    return (0);

 }
/*=========================================================================*/
/* This function selects a block                                           */
/*=========================================================================*/
int _VI_FUNC  bp2329_set_active_datalogger_block (int instrID, int mode,int block,char* name)
{
    int len,pos;
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
    
    if (bp2329_invalid_integer_range (mode, 0, 1,  -2) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (block, 0, 31,  -3) != 0)
      return bp2329_err;
    if((len = strlen(name)) > 10)
    {
     bp2329_err = -4;
     return bp2329_err; 
    }
	if(mode == 0)
	{
     Fmt (cmd, "%s<DAT:SEL:BLOCK %i",block);
     if (bp2329_send_scpi (instrID, cmd) != 0)
      return bp2329_err;
	}
	else
	{
	 while((pos = FindPattern (name, 0, len-1, " ", 0, 0)) != -1)
	  name[pos] = '_';   /* space to "_"*/
	
     Fmt (cmd, "%s<DAT:SEL:NAME %s",name);
     if (bp2329_send_scpi (instrID, cmd) != 0)
      return bp2329_err;
	}
    return (0);
}
/*=========================================================================*/
/* This function selects a block                                           */
/*=========================================================================*/
int _VI_FUNC  bp2329_get_active_datalogger_block (int instrID, int* block)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
    
	Fmt (cmd, "%s<DAT:SEL:BLOCK?");
    if (bp2329_receive_scpi (instrID, cmd,buf) != 0)
      return bp2329_err;
    Scan(buf,"%s>%d",block);  
    
    return (0);
}

/*=========================================================================*/
/* This function reads the number of values in a block                     */
/*=========================================================================*/
int _VI_FUNC  bp2329_datalogger_read_num_counts (int instrID,  int block,int *count)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
    if (bp2329_invalid_integer_range (block, 0, 31,  -2) != 0)
      return bp2329_err;
     
    Fmt(cmd,"%s<DAT:COUNT? %d",block);
    if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
      return bp2329_err;
  
	Scan(buf,"%s>%d",count); 

    return (0);
}
 
 
/*=========================================================================*/
/* This function reads 1 memory from a block                                */
/*=========================================================================*/ 
 int _VI_FUNC  bp2329_datalogger_read_block (int instrID,int block, int start_num,
                                    char * value,char* date,char * time) 
 {
	char day[3],month[3],year[5],hours[3],minutes[3],seconds[3],mseconds[3];
 
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
    if (bp2329_invalid_integer_range (block, 0, 31,  -2) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (start_num, 1, 20000,  -3) != 0)
      return bp2329_err;

    Fmt(cmd,"%s<dat:data:all? %d,%d",block,start_num);
	if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
      return bp2329_err;
  
	Scan(buf,"%s>%s[t44],%s[t44],%s[t44],%s[t44],%s[t44],%s[t44],%s[t44],%s[t44]",
	              value,
	              day,month,year,
	              hours,minutes,seconds,mseconds);
	Fmt(date,"%s<%s.%s.%s",day,month,year);
	Fmt(time,"%s<%s:%s:%s:%s",hours,minutes,seconds,mseconds);
	return(0);
}
/*=========================================================================*/
/* This function reads the evaluation data of a block                      */
/*=========================================================================*/
int _VI_FUNC  bp2329_datalogger_read_eval_values (int instrID,int block, char* minimum,
                       char *maximum, char *meanValue,char *stdDeviation)
{
    if (bp2329_invalid_integer_range (block, 0, 31,  -2) != 0)
      return bp2329_err;
    
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;

    Fmt(cmd,"%s<DAT:MIN? %d",block);
	if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
      return bp2329_err;
    strcpy(minimum,buf);  
  
    Fmt(cmd,"%s<DAT:max? %d",block);
	if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
      return bp2329_err;
  
    strcpy(maximum,buf);
    
    Fmt(cmd,"%s<DAT:AVER? %d",block);
	if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
      return bp2329_err;
      
    strcpy(meanValue,buf);  	  

    Fmt(cmd,"%s<DAT:DEV? %d",block);
	if(bp2329_receive_scpi(instrID,cmd,buf) != 0)
      return bp2329_err;
    strcpy(stdDeviation,buf);  	  
    
	return(0);
}

/*=========================================================================*/
/* This function set the output register                                   */
/*=========================================================================*/
int _VI_FUNC  bp2329_system_set_register(int instrID,int bit2,int bit1,int bit0)
{
    int status;
    bp2329_err = 0;
    
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (bit2, 0, 1,  -2) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (bit1, 0, 1,  -3) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (bit0, 0, 1,  -4) != 0)
      return bp2329_err;

	status = (bit2 * (1 << 2)) + (bit1 * (1 << 1)) + (bit0 * (1 << 0));
	
    Fmt(buf,"%s<REG:OUTP %d",status);
    if(bp2329_send_scpi(instrID,buf) != 0)
     return bp2329_err;



    return (0);
}
/*=========================================================================*/
/* This function get the input register                                    */
/*=========================================================================*/
int _VI_FUNC  bp2329_system_get_register(int instrID,int* bit3,int* bit2,int* bit1,int* bit0)
{
    int status;
    bp2329_err = 0;
    
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
      
    if(bp2329_receive_scpi(instrID,"reg:inp?",buf) != 0)
     return (bp2329_err);
    
    Scan(buf,"%s>%d",&status);

	if(status & (1 << 3))
	 *bit3 = 1;
	else 
	 *bit3 = 0;
	 
	if(status & (1 << 2))
	 *bit2 = 1;
	else 
	 *bit2 = 0;
	 
	if(status & (1 << 1))
	 *bit1 = 1;
	else 
	 *bit1 = 0;
	 
	if(status & (1 << 0))
	 *bit0 = 1;
	else 
	 *bit0 = 0;

    return (0);
}

/*=========================================================================*/
/* This function set the system Date and Time                              */
/*=========================================================================*/
int _VI_FUNC  bp2329_system_date_time(int instrID,int hh,int mm,int ss,int day,int month,
                            int year)
{
    int status;
    bp2329_err = 0;
    
    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (hh, 0, 23,  -2) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (mm, 0, 59,  -3) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (ss, 0, 59,  -4) != 0)
      return bp2329_err;
      
    if (bp2329_invalid_integer_range (day, 0, 23,  -5) != 0)
      return bp2329_err;
    if (bp2329_invalid_integer_range (month, 1, 12,  -6) != 0)
      return bp2329_err;
      
    if (bp2329_invalid_integer_range (year, 1900, 3000,  -7) != 0)
      return bp2329_err;

	
    Fmt(buf,"%s<syst:date %d,%d[w2p0],%d[w2p0]",year,month,day);
    if(bp2329_send_scpi(instrID,buf) != 0)
     return bp2329_err;
     
    Fmt(buf,"%s<syst:time %d[w2p0],%d[w2p0],%d[w2p0]",hh,mm,ss);
    if(bp2329_send_scpi(instrID,buf) != 0)
     return bp2329_err;



    return (0);
}
/*=========================================================================*/
/* This function get system error                                          */
/*=========================================================================*/
int _VI_FUNC  bp2329_system_error(int instrID)
{
   int err ,ret_error,found_error;

   bp2329_err = 0;
   if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;

   err = -1;
   ret_error = 0;
   found_error = 0;

   while(err != 0) /* Einlesen bis kein Fehler kommt */
   {
    if(bp2329_receive_scpi(instrID,":SYSTEM:ERROR?",buf) != 0)
     return (bp2329_err);

    Scan(buf,"%s>%i",&err);
/*    FmtOut("\nSys %s",buf);*/
    if((err != 0) && (found_error == 0))
    {
     found_error = 1;    /* Save only the first Error */
     ret_error = err;
    }
   }
   return (ret_error);
}

/* ========================================================================= */
/*  This function Locked or unlocked the device keyboard                     */
/* ========================================================================= */
int _VI_FUNC  bp2329_system_keyboard(int instrID,int flag)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
    return bp2329_err;
    if (bp2329_invalid_integer_range (flag, 0, 1,  -2) != 0)
    return bp2329_err;

    Fmt(buf,"%s<syst:kloc %d",flag);
    if (bp2329_send_scpi (instrID, buf) != 0)
     return bp2329_err;

    return 0;
}

/*=========================================================================*/
/* This function get execution system error                                */
/*=========================================================================*/
int _VI_FUNC  bp2329_get_exe_error(int instrID)
{
   int ret;
   int device_status;

   bp2329_err = 0;
   if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;


   device_status = 0;

   ret = bp2329_read_std_event_reg(instrID,&device_status) ;
   if(ret != 0)
    return (ret);

   if(device_status & bp2329_EXECUATION_ERROR)
   {
     ret = bp2329_system_error(instrID);
     return (ret);
   }
   return (0);
}
/*=========================================================================*/
/* This function read the Standrad event register                          */
/*=========================================================================*/
int _VI_FUNC  bp2329_read_std_event_reg(int instrID,int* status)
{
   bp2329_err = 0;
   if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;

   if(bp2329_receive_scpi(instrID,"*ESR?",buf) != 0)
    return (bp2329_err);

   Scan(buf,"%s>%i",status);

   return (0);
}
/*=========================================================================*/
/* This function read the operation event register                         */
/*=========================================================================*/
int _VI_FUNC  bp2329_read_quest_reg(int instrID,int* status)
{
   
   bp2329_err = 0;
   if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;

   if(bp2329_receive_scpi(instrID,":STAT:QUES?",buf) != 0)
    return (bp2329_err);
   /*FmtOut("\nbuf %s",buf);*/
   Scan(buf,"%s>%i",status);
   
   return (0);
}
/*=========================================================================*/
/* This function read the operation event register                         */
/*=========================================================================*/
int _VI_FUNC  bp2329_read_quest_fres_reg(int instrID,int* status)
{
   int quest_flag ;

   bp2329_err = 0;
   if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;

   if(bp2329_receive_scpi(instrID,":STAT:QUES:FRES?",buf) != 0)
    return (bp2329_err);

   Scan(buf,"%s>%i[r16]",status);
   quest_flag = *status ;
   
   if(quest_flag & bp2329_TEMP_KOMP_ERROR)  /* Bit 7 of Questionable FResistance Registers*/
   {
    bp2329_err = 310;
    return bp2329_err;
   }
   if(quest_flag & bp2329_U_CABLE_ERROR)  /* Bit 6 of Questionable FResistance Registers*/
   {
    bp2329_err = 311;
    return bp2329_err;
   }
   if(quest_flag & bp2329_I_OSCILLATION)   /* Bit 5 of Questionable FResistance Registers*/
   {
    bp2329_err = 312;
    return bp2329_err;
   }
   
   if(quest_flag & bp2329_UNDER_LOAD)      /* Bit 4 of Questionable FResistance Registers*/
   {
    bp2329_err = 313;
    return bp2329_err;
   }
   if(quest_flag & bp2329_OVER_LOAD)     /* Bit 3 of Questionable FResistance Registers*/
   {
    bp2329_err = 314;
    return bp2329_err;
   }
   if(quest_flag & bp2329_I_OVER)       /* Bit 2 of Questionable FResistance Registers*/
   {
    bp2329_err = 315;
    return bp2329_err;
   }
   if(quest_flag & bp2329_MU_OVER )    /* Bit 1 of Questionable FResistance Registers*/
   {
    bp2329_err = 316;
    return bp2329_err;
   }

   if(quest_flag & bp2329_MI_OVER)  /* Bit 0 of Questionable FResistance Registers*/
   {
    bp2329_err = 317;
    return bp2329_err;
   }



   return (0);
}

/*=========================================================================*/
/* This function read the operation event register                         */
/*=========================================================================*/
int _VI_FUNC  bp2329_read_oper_event_reg(int instrID,int* status)
{
   bp2329_err = 0;
   if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;

   if(bp2329_receive_scpi(instrID,"S:O:C?",buf) != 0)
    return (bp2329_err);

	
   Scan(buf,"%s>%i",status);

   return (0);
}
/* ========================================================================= */
/*  This function switch the calibration state ON/OFF                        */
/* ========================================================================= */
int _VI_FUNC  bp2329_cal_state(int instrID,int state)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
    return bp2329_err;
    if (bp2329_invalid_integer_range (state, 0, 1,  -2) != 0)
    return bp2329_err;

    if (bp2329_send_scpi (instrID, cal_state[state]) != 0)
     return bp2329_err;

    return 0;
}
/* ========================================================================= */
/*  This function calibration the four Wire Ranges                           */
/* ========================================================================= */
int _VI_FUNC  bp2329_cal_fres(int instrID,int range,double value)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
    if (bp2329_invalid_integer_range (range, 0, 6,  -2) != 0)
     return bp2329_err;

	Fmt(buf,"%s<%s,%f",cal_range[range],value);
    if (bp2329_send_scpi (instrID, buf) != 0)
     return bp2329_err;

    return 0;
}
/* ========================================================================= */
/*  This function calibration the 20mV Low and high                          */
/* ========================================================================= */
int _VI_FUNC  bp2329_cal_20mV_mode(int instrID, int mode)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
     
    if (bp2329_invalid_integer_range (mode, 0, 1,  -2) != 0)
     return bp2329_err;

	
    if (bp2329_send_scpi (instrID, cal_20mV_mode[mode]) != 0)
     return bp2329_err;

    return 0;
}
/* ========================================================================= */
/*  This function calibration the 20mV initalize                             */
/* ========================================================================= */
int _VI_FUNC  bp2329_cal_20mV_init(int instrID)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
     
    if (bp2329_send_scpi (instrID, "CAL:20MV:INIT") != 0)
     return bp2329_err;

    return 0;
}
/* ========================================================================= */
/*  This function calibration the 20mV initalize                             */
/* ========================================================================= */
int _VI_FUNC  bp2329_cal_20mV(int instrID,double r_value,double u1_value,double u2_value)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
    
    Fmt(buf,"%s<CAL:20MV %f,%f,%f",r_value,u1_value,u2_value) ;
    if (bp2329_send_scpi (instrID, buf) != 0)
     return bp2329_err;

    return 0;
}

/* ========================================================================= */
/*  This function calibration the Voltaeg Low and high                       */
/* ========================================================================= */
int _VI_FUNC  bp2329_cal_voltage_mode(int instrID, int mode)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
     
    if (bp2329_invalid_integer_range (mode, 0, 1,  -2) != 0)
     return bp2329_err;

	
    if (bp2329_send_scpi (instrID, cal_voltage_mode[mode]) != 0)
     return bp2329_err;

    return 0;
}
/* ========================================================================= */
/*  This function calibration the Voltage initalize                          */
/* ========================================================================= */
int _VI_FUNC  bp2329_cal_voltage_init(int instrID)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
     
    if (bp2329_send_scpi (instrID, "CAL:VOLT:INIT") != 0)
     return bp2329_err;

    return 0;
}
/* ========================================================================= */
/*  This function calibration the voltage                                    */
/* ========================================================================= */
int _VI_FUNC  bp2329_cal_voltage(int instrID,double u1_value,double u2_value)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
    
    Fmt(buf,"%s<CAL:VOLT %f,%f",u1_value,u2_value) ;
    if (bp2329_send_scpi (instrID, buf) != 0)
     return bp2329_err;

    return 0;
}
/* ========================================================================= */
/*  This function calibration the voltage                                    */
/* ========================================================================= */
int _VI_FUNC  bp2329_cal_current(int instrID,double value)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
    
    Fmt(buf,"%s<CAL:CURR %f",value) ;
    if (bp2329_send_scpi (instrID, buf) != 0)
     return bp2329_err;

    return 0;
}

/* ========================================================================= */
/*  This function calibration the PT100 Low and high                         */
/* ========================================================================= */
int _VI_FUNC  bp2329_cal_pt100_mode(int instrID, int mode,double value)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
     
    if (bp2329_invalid_integer_range (mode, 0, 1,  -2) != 0)
     return bp2329_err;

    Fmt(buf,"%s<%s %f",cal_pt100_mode[mode],value) ;	
    if (bp2329_send_scpi (instrID, buf) != 0)
     return bp2329_err;

    return 0;
}
/* ========================================================================= */
/*  This function calibration the PT100                                      */
/* ========================================================================= */
int _VI_FUNC  bp2329_cal_pt100(int instrID)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return bp2329_err;
    
    Fmt(buf,"%s<CAL:PT100") ;
    if (bp2329_send_scpi (instrID, buf) != 0)
     return bp2329_err;

    return 0;
}
int _VI_FUNC  bp2329_get_cal_status(int instrID,int* status)
{
	  int cal_status;

	  bp2329_err = 0;

	  if(bp2329_receive_scpi(instrID,"STAT:OPER:COND?",buf) != 0)
	   return bp2329_err;

	  /*FmtOut("\n Stat oper :%s ",buf);*/

	  Scan(buf,"%s>%i",&cal_status);
	  if(cal_status & bp2329_DEVICE_CALIBRATION)
	   *status = 1;
	  else
	   *status = 0;

	  return bp2329_err;
}

/*=========================================================================*/
/* This function SendSCPI                                                  */
/*=========================================================================*/
int _VI_FUNC  bp2329_send_scpi(int instrID,char* send_buffer)
{
   bp2329_err = 0;

   Fmt(cmd,"%s<%s\n",send_buffer);


   if(interface == bp2329_GPIB)  /* GPIB ?? */
   {
    bp2329_err = bp2329_write_msg_gpib(instrID,cmd,NumFmtdBytes ());

    if(bp2329_err != 0)
     return (bp2329_err);
   }
   else    /* Serial */
   {
    bp2329_err = bp2329_send_rs232(instrID,cmd);
    if(bp2329_err != 0)
     return (bp2329_err);
   }

   return(0);
}
/*=========================================================================*/
/* This function receive SCPI                                              */
/*=========================================================================*/
int _VI_FUNC  bp2329_receive_scpi(int instrID,char* send_buffer,char* receive_buffer)
{
   
   bp2329_err = 0;

   Fmt(cmd,"%s<%s\n",send_buffer);



   if(interface == bp2329_GPIB)  /* GPIB ?? */
   {
    bp2329_err = bp2329_write_msg_gpib(instrID,cmd,NumFmtdBytes ());
    if(bp2329_err != 0)
     return (bp2329_err);

    bp2329_err = bp2329_read_msg_gpib(instrID,receive_buffer,bp2329_MAX_CMD);
    if(bp2329_err != 0)
     return (bp2329_err);
   }
   else
   {
     
    bp2329_err = bp2329_receive_rs232(instrID,cmd,receive_buffer);
    if(bp2329_err != 0)
     return (bp2329_err);
     
   }


   return(0);
}
/* ========================================================================= */
/*  This function closes the instrument.                                     */
/* ========================================================================= */
int _VI_FUNC  bp2329_close (int instrID)
{
    bp2329_err = 0;

    if (bp2329_invalid_integer_range (instrID, 1, bp2329_MAX_INSTR,  -1) != 0)
     return (bp2329_err);

    if(interface == bp2329_GPIB)  /* GPIB ?? */
    {
     if (bp2329_device_closed_gpib (instrID) != 0)
      return (bp2329_err);
     bp2329_close_instr_gpib (instrID);
    }
    else
    {
     bp2329_close_instr_serial (instrID);
    }
    return (bp2329_err);
}
/* = GPIB UTILITY ROUTINES == ============================================== */
/* ========================================================================= */
/*  This function locates and initializes an entry in the Instrument Table   */
/*  and the GPIB device table for the instrument.  If successful, the        */
/*  instrument ID is returned, else a -1 is returned.  The size of the       */
/*  Instrument Table can be changed in the include file by altering the      */
/*  constant bp2329_MAX_INSTR.                                               */
/* ========================================================================= */
int _VI_FUNC  bp2329_open_instr_gpib(int addr)
{
    int i;
    int instrID;

    instrID = 0;
    /*  Check to see if the instrument is already in the Instrument Table.  */
    for (i = 1; i <= bp2329_MAX_INSTR; i++)
    if (address[i] == addr)
    {
        instrID = i;
        i = bp2329_MAX_INSTR;
    }
    /*  If it is not in the instrument table, open an entry for the instrument.  */
    if (instrID <= 0)
    for (i = 1; i <= bp2329_MAX_INSTR; i++)
        if (!address[i])
        {
        instrID = i;
        i = bp2329_MAX_INSTR;
        }
    /*  If an entry could not be opened in the Instrument Table, return an error. */
    if (instrID <= 0)
    {
    bp2329_err = 220;
    return  -1;
    }
    /*   If the device has not been opened in the GPIB device table (bd[ID] = 0), */
    /*   then open it.                                                            */
    if (bd[instrID] <= 0)
    {
    if (instr_cnt <= 0)
        CloseInstrDevs ("bp2329");
    bd[instrID] = OpenDev ("", "bp2329");
    if (bd[instrID] <= 0)
    {
        bp2329_err = 220;
        return  -1;
    }
    }
    /*   Change the primary address of the device     */
    if (ibpad (bd[instrID], addr) < 0)
    {
    bp2329_err = 233;
    return  -1;
    }
    address[instrID] = addr;
    instr_cnt++;
    return instrID;
}

/* ========================================================================== */
/*  This function closes the instrument by removing it from the GPIB device   */
/*  table and setting the address and the bd to zero in the Instrument Table. */
/*  The return value is equal to the global error variable.                   */
/* ========================================================================== */
int _VI_FUNC  bp2329_close_instr_gpib (int instrID)
{

    if (bd[instrID] != 0) 
    {
    CloseDev (bd[instrID]);
    bd[instrID] = 0;
    address[instrID] = 0;
    instr_cnt--;
    }
    else
    bp2329_err = 221;
    return bp2329_err;
}
/* ========================================================================= */
/*  This function checks to see if the module has been initialized.  If the  */
/*  device has not been opened, a 1 is returned, 0 otherwise.                */
/* ========================================================================= */
int _VI_FUNC  bp2329_device_closed_gpib (int instrID)
{

    if (bd[instrID] <= 0)
    {
    bp2329_err = 232;
    return  -1;
    }
    return 0;
}

/* ========================================================================= */
/*  This function reads a buffer of data from the instrument. The return     */
/*  value is equal to the global error variable.                             */
/* ========================================================================= */
int _VI_FUNC  bp2329_read_msg_gpib(int instrID,char *msg,int cnt)
{
    ibrd (bd[instrID], msg, (long)cnt);
    
    if(iberr != 0)
        bp2329_err = 231;
    else
        bp2329_err = 0;

	if(ibcntl >= 2)
     msg[ibcntl-2] = 0x00;
     
    /*FmtOut("\n%s",msg);*/

    return bp2329_err;
}

/* ========================================================================= */
/*  This function writes a buffer of data to the instrument. The return      */
/*  value is equal to the global error variable.                             */
/* ========================================================================= */
int _VI_FUNC  bp2329_write_msg_gpib (int instrID,char *msg,int cnt)
{
/*    FmtOut("\n%s",msg);*/
	
	ibwrt (bd[instrID], msg, (long)cnt);
	
    if (iberr != 0)
     bp2329_err = 230;
    else
     bp2329_err = 0;
     
    return bp2329_err;
}
/* = SERIAL UTILITY ROUTINES ============================================== */
/* ========================================================================= */
/*  This function locates and initializes an entry in the Instrument Table   */
/*  and the device table for the instrument.  If successful, the            */
/*  instrument ID is returned, else a -1 is returned.  The size of the       */
/*  Instrument Table can be changed in the include file by altering the      */
/*  constant bp2329_MAX_INSTR.                                               */
/* ========================================================================= */
/*=========================================================================*/
/* Function: Open Instrument                                               */
/* Purpose:  This function opens and configures the selected COM port      */
/*           The return value is equal to the global error variable.       */
/*=========================================================================*/
int _VI_FUNC  bp2329_open_instr_serial (int com_p, long baud_r, double timout)
{
    short i, instrID;
    instrID = 0;
	
	bp2329_err = 0;
    /*  Check to see if the instrument is already in the Instrument Table.  */
    for (i = 1; i <= bp2329_MAX_INSTR; i++)
    {
		if (dev_port[i] == com_p)
        {
            instrID = i;
            i = bp2329_MAX_INSTR;
        }
    }   
    /*  If it is not in the instrument table, open an entry for the instrument  */
    if (instrID <= 0)
        for (i = 1; i <= bp2329_MAX_INSTR; i++)
        {
			if (!dev_port[i])
            {
                instrID = i;
                i = bp2329_MAX_INSTR;
            }
         }   
    /*  If an entry could not be opened in the Instrument Table, return an error */
    if (instrID <= 0)  
    {
        bp2329_err = 220 ;
        return -1;
    }

	/* If the device has not been opened in the RS232 device table           */
	/* (port[ID] = 0), then open a com port for the instrument module.       */
	/* But first make sure the Com-port is closed.                           */
	
    if (dev_port[instrID] <= 0) 
    {
        if (instr_cnt > 0) 
        {
            CloseCom (com_p);
            if ((rs232err != 0) && (rs232err != -3))
            {
                bp2329_err = rs232err ;
                return -1;
            }    
	}

	OpenComConfig (com_p, "", baud_r, 0, 8, 1, 512, 512);
        if (rs232err != 0)
        {
            bp2329_err = rs232err ;
            return -1;
        }

        instr_cnt += 1;
        dev_port[instrID] = com_p;
     }

/*  Set the timeout value for the instrument.  If an error occurs, set  */


    SetComTime (com_p, timout);
    if (rs232err != 0) 
    {
        bp2329_err = rs232err;
        return -1;
    }

    
    return instrID;
}


/*=========================================================================*/
/* Function: Close Instrument Serial                                       */
/* Purpose:  This function closes the port for the instrument module and   */
/*           sets the port to zero. The return value is equal to the       */
/*           global error variable.                                        */
/*=========================================================================*/
int _VI_FUNC  bp2329_close_instr_serial (int instrID)
{
/*  Close the com port.  If error, set bp2329_err = rs232err  */
    if (dev_port[instrID] != 0)  
    {
        CloseCom (dev_port[instrID]);
        if (rs232err != 0)
            bp2329_err = rs232err;
        dev_port[instrID] = 0;
        port_addr[instrID] = 0;
        instr_cnt -= 1;
    }
    else
        bp2329_err = 221;

    return bp2329_err;
}

/*=========================================================================*/
/* This function send a comand and message and wait for ACK                */
/*=========================================================================*/
int _VI_FUNC  bp2329_send_rs232(int instrID,char* msg)
{
    int ret,r_byte,l;
    

    FlushInQ(dev_port[instrID]);           /* Clear Port */
    
    Fmt(out_buf,"%c%s%c",STX,msg,ETX);     /* String set*/
    
    l = StringLength(out_buf);
    ret = ComWrt(dev_port[instrID], out_buf, l);     /* Write the Message */
    if (rs232err != 0)
     return (230);


    r_byte = ComRdByte(dev_port[instrID]);			/* read the answer of slave*/
    if (rs232err != 0)
     return (231);

	/*FmtOut("\nsend %s ret %d \n",out_buf,r_byte);*/
    
    switch(r_byte)					/*Witch Answer ?*/
    {
      case ACK:
       return  (0);					/* ACK => all ready*/
       break;
      case NAK:
       return (300);                /* NAK Error-Code */
       break;
      case EOT:						
       return (301);				/*EOT Error Code*/
       break;
    }

    return (0);
}

/*=========================================================================*/
/* This function receive a string            							   */
/*=========================================================================*/
int _VI_FUNC  bp2329_receive_rs232(int instrID,char *cmd,char* receive_buf)
{
    int ret;

	/*FmtOut("\n++send '%s'",cmd);*/
    bp2329_err = bp2329_send_rs232(instrID,cmd);
    if(bp2329_err != 0)
     return (bp2329_err);
     
    FlushInQ(dev_port[instrID]);           /* Clear Port */
    
    ret = bp2329_receive_rs232_message(instrID,receive_buf);
    if(ret != 0)
     return (ret);
    
     
    return(0);              				 /* 0 = ACK == OK*/
	
}
/*=========================================================================*/
/* This function receive a string            							   */
/*=========================================================================*/
int _VI_FUNC  bp2329_receive_rs232_message(int instrID,char* receive_buf)
{
    int first_byte , r_byte,l;

	ComWrtByte(dev_port[instrID],EOT);/* send EOT to slave => Master is ready to read */
	
    first_byte = ComRdByte(dev_port[instrID]);   /* Read the EOT or STX from Port*/
    /*FmtOut("\n**First '%d'",first_byte);*/
    if(first_byte == EOT)
    {
   	/*FmtOut("\n**First '%d'",first_byte);*/
     	return (301);
    } 

    r_byte = ComRdTerm(dev_port[instrID],in_buf,bp2329_MAX_CMD-1,ETX);/* Read to ETX*/
    if (rs232err != 0)
     return (231);
 
    in_buf[r_byte] = 0x00;				 /* terminate the string*/
    
    l = StringLength(in_buf);     
    
    /*FmtOut("\n++ inbuf '%s'",in_buf);*/
    
    CopyString (receive_buf, 0,in_buf,0,l-2); /* Copy without LF,ETX*/
    ComWrtByte(dev_port[instrID],ACK);	 /* Send ACK to slave => all ready */
    
    r_byte = ComRdByte(dev_port[instrID]);   /* Read the EOT from Port => End of Transmission*/
    /*FmtOut("\n**last '%c'",r_byte);*/
    if(r_byte == EOT)
     return (0);							 /* No more Datas to read*/
    /*else
     return (309);							  /* More Datas  to read*/
    
    return(0);   
}

/* = Global UTILITY ROUTINES ==============================================*/
/*=========================================================================*/
/* Function: Invalid Integer Range                                         */
/* Purpose:  This function checks an integer to see if it lies between a   */
/*           minimum and maximum value.  If the value is out of range, set */
/*           the global error variable to the value err_code.  If the      */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int _VI_FUNC  bp2329_invalid_integer_range (int val,int min,int max,int err_code)
{
  if ((val < min) || (val > max))
  {
    bp2329_err = err_code;
    return -1;
  }
  return 0;
}
/*=========================================================================*/
/* Function: Invalid Real Range                                            */
/* Purpose:  This function checks a real number to see if it lies between  */
/*           a minimum and maximum value.  If the value is out of range,   */
/*           set the global error variable to the value err_code.  If the  */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int _VI_FUNC  bp2329_invalid_real_range (double val,double min,double max,int err_code)
{
  if ((val < min) || (val > max))
  {
    bp2329_err = err_code;
    return -1;
  }
  return 0;
}
void bp2329_setup_arrays (void)
{
    int i;

    if(!setup_first)/*----------- Reset the Arrays before initial --------*/
    {
     for (i=1; i<= bp2329_MAX_INSTR; i++)
     {
       dev_port[i]  =  0;
       address[i] = 0;
     }
     setup_first = 1;
     instr_cnt = -1;
    }
    port_addr[0] = 0x3f8;
    port_addr[1] = 0x2f8;
    port_addr[2] = 0x3e8;
    port_addr[3] = 0x2e8;
    port_addr[4] = 0x2F0;
    port_addr[5] = 0x3E0;   
    port_addr[6] = 0x2E0;
    port_addr[7] = 0x260;



    int_level[0] = 4;
    int_level[1] = 3;
    int_level[2] = 4;
    int_level[3] = 3;

	sense_aver_ctrl[0] = "SENSE:AVER:TCON MOV" ;
    sense_aver_ctrl[1] = "SENSE:AVER:TCON REP"	;
    
    sense_loading[0] = "sens:fres:load real";       /* Ohmic*/
    sense_loading[1] = "sens:fres:load comp";       /* Ohmic*/

    sense_meas_mode[0] = "init:cont off";
    sense_meas_mode[1] = "init:cont on";
    
    sense_limit_20mv[0] = "sour:volt:lim:stat off";    
    sense_limit_20mv[1] = "sour:volt:lim:stat on";
    
    sense_meas_sequence[0] = "sens:fres:mode refc";
    sense_meas_sequence[1] = "sens:fres:mode nonc";
    sense_meas_sequence[2] = "sens:fres:mode onec";
    sense_meas_sequence[3] = "sens:fres:mode stan";
    
    sense_num_of_cycles[0] = "sens:fres:nplc min";
    sense_num_of_cycles[1] = "sens:fres:nplc med";
    sense_num_of_cycles[2] = "sens:fres:nplc max";
    sense_num_of_cycles[3] = "sens:fres:nplc stan";

     
	
    sense_range[0] = ":SENSE:FRES:RANG:AUTO ON";
    sense_range[1] = ":SENSE:FRES:RANG:MAN 200MOHM";
    sense_range[2] = ":SENSE:FRES:RANG:MAN 2OHM";
    sense_range[3] = ":SENSE:FRES:RANG:MAN 20OHM";
    sense_range[4] = ":SENSE:FRES:RANG:MAN 200OHM";
    sense_range[5] = ":SENSE:FRES:RANG:MAN 2KOHM";
    sense_range[6] = ":SENSE:FRES:RANG:MAN 20KOHM";
    sense_range[7] = ":SENSE:FRES:RANG:MAN 200KOHM";
    
    sense_lower_range[1] = ":SENSE:FRES:RANG:LOW 200MOHM";
    sense_lower_range[2] = ":SENSE:FRES:RANG:LOW 2OHM";
    sense_lower_range[3] = ":SENSE:FRES:RANG:LOW 20OHM";
    sense_lower_range[4] = ":SENSE:FRES:RANG:LOW 200OHM";
    sense_lower_range[5] = ":SENSE:FRES:RANG:LOW 2KOHM";
    sense_lower_range[6] = ":SENSE:FRES:RANG:LOW 20KOHM";
    sense_lower_range[7] = ":SENSE:FRES:RANG:LOW 200KOHM";
    
    sense_upper_range[1] = ":SENSE:FRES:RANG:UPP 200MOHM";
    sense_upper_range[2] = ":SENSE:FRES:RANG:UPP 2OHM";
    sense_upper_range[3] = ":SENSE:FRES:RANG:UPP 20OHM";
    sense_upper_range[4] = ":SENSE:FRES:RANG:UPP 200OHM";
    sense_upper_range[5] = ":SENSE:FRES:RANG:UPP 2KOHM";
    sense_upper_range[6] = ":SENSE:FRES:RANG:UPP 20KOHM";
    sense_upper_range[7] = ":SENSE:FRES:RANG:UPP 200KOHM";

    display_mode[0] = "calc:math:expr OHM";
    display_mode[1] = "calc:math:expr DELT";
    display_mode[2] = "calc:math:expr LIM";
    
    display_resolution[0] = "sens:fres:res 0.0005" ;
    display_resolution[1] = "sens:fres:res 0.00005" ;
     
    calc_limit_fault[0] = "calc:lim:faul none";
    calc_limit_fault[1] = "calc:lim:faul upp";

    sense_tcomp[0] = "sens:tcom MAN";
    sense_tcomp[1] = "sens:tcom PT100";
    sense_tcomp[2] = "sens:tcom UINP";
    sense_tcomp[3] = "sens:tcom IINP";
    
    cal_state[0] = "CAL:STAT OFF";
    cal_state[1] = "CAL:STAT ON";
    
    cal_range[0] = "CAL:FRES 200MOHM";
	cal_range[1] = "CAL:FRES 2OHM"	  ;
	cal_range[2] = "CAL:FRES 20OHM"   ;
	cal_range[3] = "CAL:FRES 200OHM"  ;
	cal_range[4] = "CAL:FRES 2KOHM"   ;
	cal_range[5] = "CAL:FRES 20KOHM"  ;
	cal_range[6] = "CAL:FRES 200KOHM" ;
	
    cal_20mV_mode[0] = "CAL:20MV:LOW";
    cal_20mV_mode[1] = "CAL:20MV:HIGH";
    
    cal_voltage_mode[0] = "CAL:VOLT:LOW";
    cal_voltage_mode[1] = "CAL:VOLT:HIGH";

    cal_pt100_mode[0] = "CAL:PT100:LOW";
    cal_pt100_mode[1] = "CAL:PT100:HIGH";
    
}
double convert_string_to_value(char * string)
{
	double value;
	int l;
	
	l = strlen(string);
	
	if( FindPattern (string, 0, l, "MOHM", 0, 0) > 0)		   /* mOhm*/
	 value = atof(string) / 1000.0;
	else if ( FindPattern (string, 0, l, "KOHM", 0, 0) > 0) 		/* kOhm*/
	 value =  atof(string) * 1000.0;
	else 
	 value =  atof(string);
	 
	return value;
}	 

/*= THE END ===============================================================*/
